#include <string.h>
#include "wm_include.h"

#include "ooff.h"
#include "my_vfs_fat.h"

#define FS_API_DEBUG                0

#define FS_API_SUPPORT_STD_IO       1

#if FS_API_DEBUG
#define FS_API_DEBUG_PRINT          printf
#else
#define FS_API_DEBUG_PRINT(...)
#endif

#if FS_API_SUPPORT_STD_IO
extern int sendchar(int ch);
static int fs_api_write(const void *ptr, size_t len)
{
	size_t i;
	char *p;

	p = (char *)ptr;
	
	for (i = 0; i < len; i++) 
	{
		(void)sendchar(*p++);
	}

	return len;
}
#endif

FILE *fopen(const char *restrict pathname, const char *restrict mode)
{
    FS_API_DEBUG_PRINT("\n%s: [%s] [%s]\n\n", __func__, pathname, mode);
    fs_user_mount_t *vfs_fat;
    FRESULT res;
    FIL *_file;
    BYTE _mode;
    BYTE r = 0;
    BYTE w = 0;
    BYTE a = 0;
    BYTE plus = 0;

    vfs_fat = my_vfs_fat_get_ctx();
	if (!vfs_fat)
	{
	    printf("%s: Couldn't find vfs fat!\n", __func__);
        return NULL;
    }

    _file = tls_mem_alloc(sizeof(FIL));
    if (!_file)
	{
	    printf("%s: mem err!\n", __func__);
        return NULL;
    }

    _mode = 0;
    if (strchr(mode, 'r'))
        r = 1;
    if (strchr(mode, 'w'))
        w = 1;
    if (strchr(mode, 'a'))
        a = 1;
    if (strchr(mode, '+'))
        plus = 1;

    if (r)
        _mode |= FA_READ;
    if (w)
        _mode |= FA_WRITE | FA_CREATE_ALWAYS;
    if (a)
        _mode |= FA_WRITE | FA_CREATE_NEW | FA_OPEN_APPEND;
    if (r && plus)
        _mode |= FA_READ | FA_WRITE;
    if (w && plus)
        _mode |= FA_WRITE | FA_CREATE_ALWAYS;
    if (a && plus)
        _mode |= FA_READ | FA_WRITE | FA_CREATE_NEW | FA_OPEN_APPEND;

    res = f_open(&vfs_fat->fatfs, _file, pathname, _mode);
	if (res != FR_OK)
	{
		printf("%s: Could open %s\n", __func__, pathname);
		tls_mem_free(_file);
		return NULL;
	}

    FS_API_DEBUG_PRINT("%s: opened file %p\n", __func__, _file);

    return (FILE *)_file;
}

int fclose(FILE *stream)
{
    FS_API_DEBUG_PRINT("\n%s: [%p]\n", __func__, stream);
    FRESULT res;

#if FS_API_SUPPORT_STD_IO
    if ((stream == stdin) || (stream == stdout) || (stream == stderr))
        return 0;
#endif

    res = f_close((FIL*)stream);
    if (res != FR_OK)
    {
        printf("close file error, res = %d\r\n", res);
        return -1;
    }
    else
    {
        tls_mem_free(stream);
        return 0;
    }
}

size_t fread(void *restrict ptr, size_t size, size_t nitems, FILE *restrict stream)
{
    FS_API_DEBUG_PRINT("\n%s: [%p] [%d-%d]\n\n", __func__, stream, size, nitems);

#if FS_API_SUPPORT_STD_IO
    if ((stream == stdin) || (stream == stdout) || (stream == stderr))
        return EOF;
#endif

#if !DRV_USE_SDIO //spi
    FRESULT res;
    FIL *_file;
    UINT bw;
    UINT btr;

    _file = (FIL *)stream;
    btr = size * nitems;

    res = f_read(_file, ptr, btr, &bw);
    if (res != FR_OK || bw != btr)
    {
        if (res == FR_OK)
        {
            printf("%s: want %d, but %d\n", __func__, btr, bw);
            if (bw % size)
                printf("%s: nitems want %d, but not align %d\n", __func__, nitems, bw % size);
            return bw / size;//bw;
        }
        else
        {
            return EOF;
        }
    }

    return nitems;//btr;
#else //sdio
    FRESULT res;
    FIL *_file;
    UINT bw;
    UINT btr;
    UINT remain;
    UINT total = 0;

    _file = (FIL *)stream;
    remain = size * nitems;

    while (remain)
    {
        if (remain > 512)
        {
            btr = 512;
            remain -= btr;
        }
        else
        {
            btr = remain;
            remain = 0;
        }

        res = f_read(_file, (unsigned char *)ptr + total, btr, &bw);
        //printf("res %d, btr %d, bw %d, remain %d, total %d, len %d\n", res, btr, bw, remain, total, size * nitems);
        if (res != FR_OK || bw != btr)
        {
            if (res == FR_OK)
            {
                printf("%s: want %d, but %d\n", __func__, btr, bw);
                total += bw;
                if (total % size)
                    printf("%s: nitems want %d, but not align %d\n", __func__, nitems, total % size);
                return total / size;//total;
            }
            else
            {
                return EOF;
            }
        }

        total += bw;
    }

    return nitems;//total / size;//total;
#endif
}

size_t fwrite(const void *restrict ptr, size_t size, size_t nitems, FILE *restrict stream)
{
    FS_API_DEBUG_PRINT("\n%s: [%p] [%d-%d]\n\n", __func__, stream, size, nitems);
    FRESULT res;
    FIL *_file;
    UINT bw;
    UINT btw;

    _file = (FIL *)stream;
    btw = size * nitems;

#if FS_API_SUPPORT_STD_IO
    if ((stream == stdin) || (stream == stdout) || (stream == stderr))
        return fs_api_write(ptr, btw);
#endif

    res = f_write(_file, ptr, btw, &bw);
    if (res != FR_OK || bw != btw)
    {
        printf("%s: ret %d, want %d, but %d\n", __func__, res, btw, bw);
        return EOF;
    }

    return nitems;//btw;
}

long ftell(FILE *stream)
{
    FS_API_DEBUG_PRINT("\n%s: [%p]\n\n", __func__, stream);

    return f_tell((FIL *)stream);
}

int fseek(FILE *stream, long offset, int whence)
{
    FS_API_DEBUG_PRINT("\n%s: [%p] [%ld-%d]\n\n", __func__, stream, offset, whence);
    FRESULT res = FR_INVALID_PARAMETER;
    FIL *_file;

    _file = (FIL *)stream;

    if (SEEK_SET == whence)
    {
        res = f_lseek(_file, offset);
    }
    else if (SEEK_CUR == whence)
    {
        FSIZE_t curr = f_tell(_file);
        res = f_lseek(_file, curr + offset);
    }
    else if (SEEK_END == whence)
    {
        FSIZE_t fsize = f_size(_file);
        res = f_lseek(_file, fsize - offset - 1);
    }

    return res == FR_OK ? 0 : -1;
}

#if 1
char *fgets(char *restrict s, int n, FILE *restrict stream)
{
    FS_API_DEBUG_PRINT("\n%s: [%p] [%d]\n\n", __func__, stream, n);
    FRESULT res;
    FIL *_file;
    UINT bw;
    UINT off = 0;

#if FS_API_SUPPORT_STD_IO
    if ((stream == stdin) || (stream == stdout) || (stream == stderr))
        return NULL;
#endif

    n--;//for tail '\0'
    _file = (FIL *)stream;
    while (n)
    {
        res = f_read(_file, s + off, 1, &bw);
        if (res != FR_OK || bw != 1)
        {
            if (res == FR_OK)
            {
                /* EOF */
                *(char *)(s + off) = '\0';
                return s;
            }
            else
            {
               printf("%s: read err ret %d\n", __func__, res);
               return NULL;
            }
        }

        if ('\n' == *(char *)(s + off))
        {
            *(char *)(s + off + 1) = '\0';
            return s;
        }

        off++;
        n--;
    }

    *(char *)(s + off) = '\0';

    return s;
}

int fputs(const char *restrict s, FILE *restrict stream)
{
    FS_API_DEBUG_PRINT("\n%s: [%p]\n\n", __func__, stream);
    FRESULT res;
    FIL *_file;
    UINT bw;
    UINT btw;

    _file = (FIL *)stream;
    btw = strlen(s);

#if FS_API_SUPPORT_STD_IO
    if ((stream == stdin) || (stream == stdout) || (stream == stderr))
        return fs_api_write(s, btw);
#endif

    res = f_write(_file, s, btw, &bw);
    if (res != FR_OK || bw != btw)
    {
        printf("%s: ret %d, want %d, but %d\n", __func__, res, btw, bw);
        return EOF;
    }

    return btw;
}
#endif

#if 0 //conflict libc_port.c
int fgetc(FILE *stream)
{
    printf("\n%s: [%p]\n\n", __func__, stream);
    FRESULT res;
    FIL *_file;
    UINT bw;
    UINT btr;
    int rchar;

#if FS_API_SUPPORT_STD_IO
    if ((stream == stdin) || (stream == stdout) || (stream == stderr))
        return EOF;
#endif

    _file = (FIL *)stream;
    btr = 1;

    res = f_read(_file, &rchar, btr, &bw);
    if (res != FR_OK || bw != btr)
    {
        return EOF;
    }

    return btr;
}

int fputc(int c, FILE *stream)
{
    printf("\n%s: [%p]\n\n", __func__, stream);
    FRESULT res;
    FIL *_file;
    UINT bw;
    UINT btw;

#if FS_API_SUPPORT_STD_IO
    if ((stream == stdin) || (stream == stdout) || (stream == stderr))
        return fs_api_write(&c, 1);
#endif

    _file = (FIL *)stream;
    btw = 1;

    res = f_write(_file, &c, btw, &bw);
    if (res != FR_OK || bw != btw)
    {
        printf("%s: ret %d, want %d, but %d\n", __func__, res, btw, bw);
        return EOF;
    }

    return btw;
}

int getc(FILE *stream)
{
    return fgetc(stream);
}

int putc(int c, FILE *stream)
{
    return fputc(c, stream);
}
#endif

int ferror(FILE *stream)
{
    FS_API_DEBUG_PRINT("\n%s: [%p]\n\n", __func__, stream);

    return f_error((FIL *)stream);
}

int feof(FILE *stream)
{
    FS_API_DEBUG_PRINT("\n%s: [%p]\n\n", __func__, stream);

    return f_eof((FIL *)stream);
}

void rewind(FILE *stream)
{
    FS_API_DEBUG_PRINT("\n%s: [%p]\n\n", __func__, stream);

    f_rewind((FIL *)stream);
}
