#include <stdint.h>
#include "wm_include.h"
#include "wm_pmu.h"
#include "wm_gpio_afsel.h"

#define DEMO_UART_NO    TLS_UART_1

/********************************************************/
#define DFPLAYER_SEND_LENGTH 10

#define Stack_Header 0
#define Stack_Version 1
#define Stack_Length 2
#define Stack_Command 3
#define Stack_ACK 4
#define Stack_Parameter 5
#define Stack_CheckSum 7
#define Stack_End 9

static uint8_t _sending[DFPLAYER_SEND_LENGTH] = {0x7E, 0xFF, 06, 00, 01, 00, 00, 00, 00, 0xEF};

static uint16_t calculateCheckSum(uint8_t *buffer)
{
  uint16_t sum = 0;
  for (int i=Stack_Version; i<Stack_CheckSum; i++) {
    sum += buffer[i];
  }
  return -sum;
}

static void uint16ToArray(uint16_t value, uint8_t *array)
{
  *array = (uint8_t)(value>>8);
  *(array+1) = (uint8_t)(value);
}

static void sendStack(uint8_t command, uint16_t argument)
{
  _sending[Stack_Command] = command;
  uint16ToArray(argument, _sending+Stack_Parameter);
  uint16ToArray(calculateCheckSum(_sending), _sending+Stack_CheckSum);

  tls_uart_write(DEMO_UART_NO, (char *)_sending, DFPLAYER_SEND_LENGTH);
  tls_os_time_delay(100);
}

static void volume(uint8_t volume)
{
  sendStack(0x06, volume);
}

static void enableLoopAll(void)
{
  sendStack(0x11, 0x01);
}

static void start(void)
{
  sendStack(0x0D, 0);
}

static void pause(void)
{
  sendStack(0x0E, 0);
}

static void reset(void)
{
  sendStack(0x0C, 0);
}

static void next(void)
{
  sendStack(0x01, 0);
}

static uint8_t g_play_ing = 0;
static uint8_t g_play_vol = 25;//0-30
//static int8_t g_vol_dir = -1;
/********************************************************/

#define DEMO_TASK_PRIO              32
#define DEMO_TASK_STK_SIZE          2048

#define DEMO_TASK_MSG_QUEUE_SIZE    4

#define DEMO_KEY                    WM_IO_PB_05

#define DEMO_KEY_USE_EX             0
#define DEMO_KEY_DELAY_TIME         250 /* 250tick = 500ms */

#define DEMO_MSG_KEY_PRESS          0x1

static tls_os_queue_t *demo_task_msg_queue = NULL;

static OS_STK demo_task_stk[DEMO_TASK_STK_SIZE / sizeof(OS_STK)];

static void demo_key_callback(void *ctx)
{
	u8 ret;
	enum tls_io_name pin;

	pin = (enum tls_io_name)(unsigned long)ctx;

	ret = tls_get_gpio_irq_status(pin);
	//printf("key status %hhu.\n", ret);
	if (ret) /* ʵԼio״̬޸״̬ж */
	{
		tls_clr_gpio_irq_status(pin);
        tls_os_queue_send(demo_task_msg_queue, (void *)DEMO_MSG_KEY_PRESS, 0);
	}

	return;
}

static void demo_key_event(void)
{
    int key_state = 0;
	int key_count = 0;
#if DEMO_KEY_USE_EX
	u32 curr_time;
	static u8  last_key_cnt  = 0;
	static u32 last_key_time = 0;
#endif

    do {
	    tls_os_time_delay(2);
	    key_count++;
	    key_state = tls_gpio_read(DEMO_KEY);
	} while(!key_state && (key_count <= (750 * 3))); /* ʵԼio״̬޸״̬жϣݶ3Ϊ750 * 2 tick = 1500 * 2 ms = 3s */

	//printf("key pressed %g seconds\r\n", (double)key_count * 2 * 2 / 1000);

	if ((key_count <= 250) && (key_count >= 5)) /* 20ms-1s */
	{
	    printf("key press short: playing %d.\n", g_play_ing); /* ݵİһ°ť */

	    if (g_play_ing)
	    {
            pause();
	    }
	    else
	    {
            start();
	    }
	    g_play_ing = !g_play_ing;

#if DEMO_KEY_USE_EX
        curr_time = tls_os_get_time();
        if ((curr_time - last_key_time) <= DEMO_KEY_DELAY_TIME)
        {
            if (1 == last_key_cnt)
            {
                printf("press thrid.\r\n"); /* ݵİ°ť */
            }
            else
            {
                printf("press second.\r\n"); /* ݵİ°ť */
                last_key_cnt++;
            }
        }
        else
        {
            last_key_cnt = 0;
        }

	    last_key_time = curr_time;
#endif
	}
	else if (key_count >= 750) /* >=3s */
	{
        printf("key press long.\n"); /* һ°ť */

#if 0
        if (-1 == g_vol_dir)
        {
            if (g_play_vol > 0)
            {
                g_play_vol--;
            }
            else
            {
                g_vol_dir = 1;
                g_play_vol = 1;
            }
        }
        else if (1 == g_vol_dir)
        {
            if (g_play_vol < 30)
            {
                g_play_vol++;
            }
            else
            {
                g_vol_dir = -1;
                g_play_vol = 29;
            }
        }
        volume(g_play_vol);
#else
        next();
#endif
	}

    return;
}

static void demo_key_init(void)
{
    /* ʵԼio״̬޸ */
    tls_gpio_cfg(DEMO_KEY, WM_GPIO_DIR_INPUT, WM_GPIO_ATTR_FLOATING);
  	tls_gpio_isr_register(DEMO_KEY, demo_key_callback, (void *)DEMO_KEY);
	tls_gpio_irq_enable(DEMO_KEY, WM_GPIO_IRQ_TRIG_FALLING_EDGE);

    return;
}

static void demo_uart_init(void)
{
    tls_uart_options_t opts;

    //ѡʹõżĸù
    /* UART1_RX-PB07  UART1_TX-PB06 */
    wm_uart1_rx_config(WM_IO_PB_07);
    wm_uart1_tx_config(WM_IO_PB_06);

    opts.baudrate = 9600;
    opts.charlength = TLS_UART_CHSIZE_8BIT;
    opts.flow_ctrl = TLS_UART_FLOW_CTRL_NONE;
    opts.paritytype = TLS_UART_PMODE_DISABLED;
    opts.stopbits = TLS_UART_ONE_STOPBITS;

    tls_uart_port_init(DEMO_UART_NO, &opts, 0);

    return;
}

static void demo_mp3_player_init(void)
{
    reset();

    enableLoopAll();

    pause();

    volume(g_play_vol);

    g_play_ing = 0;

    return;
}

static void demo_low_power_init(void)
{
    u8 auto_reconnect = 0xff;

	tls_wifi_auto_connect_flag(WIFI_AUTO_CNT_FLAG_GET, &auto_reconnect);
    if (auto_reconnect != WIFI_AUTO_CNT_OFF)
    {
        auto_reconnect = WIFI_AUTO_CNT_OFF;
        tls_wifi_auto_connect_flag(WIFI_AUTO_CNT_FLAG_SET, &auto_reconnect);
    }

    tls_close_peripheral_clock(TLS_PERIPHERAL_TYPE_I2C | TLS_PERIPHERAL_TYPE_UART2 | 
                               TLS_PERIPHERAL_TYPE_UART3 | TLS_PERIPHERAL_TYPE_UART4 | 
                               TLS_PERIPHERAL_TYPE_PWM | TLS_PERIPHERAL_TYPE_I2S |
                               TLS_PERIPHERAL_TYPE_BT);

    return;
}

static void demo_task_entry(void *sdata)
{
    int   ret = WM_FAILED;
	void *msg = NULL;

	demo_low_power_init();

    demo_uart_init();

    demo_mp3_player_init();

    demo_key_init();

	for( ; ; )
	{
        ret = tls_os_queue_receive(demo_task_msg_queue, (void **)&msg, 0, 0);
        if (!ret)
        {
            switch((u32)msg)
        	{
                case DEMO_MSG_KEY_PRESS:
                {
                    demo_key_event();
                    break;
                }
                default:
        		{
        			break;
                }
            }
		}
	}
}

void UserMain(void)
{
	printf("\n\n mini mp3 player demo\n\n");

    tls_os_queue_create(&demo_task_msg_queue, DEMO_TASK_MSG_QUEUE_SIZE);
    tls_os_task_create(NULL, "demo_task", demo_task_entry, NULL, (void *)demo_task_stk, DEMO_TASK_STK_SIZE, DEMO_TASK_PRIO, 0);
}

