#include <stdio.h>
#include <errno.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <signal.h>
#include <pthread.h>
#include <sys/epoll.h>
#include <arpa/inet.h>
#include <sys/socket.h>

#include "yc_opt.h"
#include "yc_basetype.h"
#include "yc_debug.h"
#include "list.h"
#include "jhash.h"
#include "yc_queue.h"
#include "yc_hash.h"

#include "wns_conf.h"

#define WNS_DEBUG   YC_DBG_PRINT

/* 
tlv 

1byte 1byte len bytes
type  len   value
*/
typedef enum {
    WNS_TLV_TYPE_VER = 0,
    WNS_TLV_TYPE_STATUS, /* 0-client, 1-resource */
    WNS_TLV_TYPE_MAC,
    WNS_TLV_TYPE_IP,
    WNS_TLV_TYPE_MASK,
    WNS_TLV_TYPE_GATE
} wns_tlv_type_e;

typedef struct {
    uchar rmac[6]; /* resource mac */
    uchar cmac[6]; /* client mac */

    uint ip;
    uint mask;
    uint gate;
} wns_conf_s;

typedef struct {
    uint index;
    int state; /* -1-not used, 0-wait auth, 1-forward */
} wns_node_ctx_s;

static uint g_wns_conf_size = 0;
static wns_conf_s *g_wns_conf_table = NULL;
static wns_node_ctx_s g_wns_socket_table[WNS_EPOLL_SIZE] = {-1};

static int g_wns_thread_queue[WNS_THREAD_NUM] = {-1};

static int g_wns_server_fd = -1;

static int wns_load_conf(void)
{
    int i = 0;
    FILE *conf;
    char confline[32];
    char *sep;

    conf = fopen(WNS_CONF_FILE, "rb");
    if (!conf)
    {
        YC_DBG_PRINT("open conf file \"%s\" error.\r\n", WNS_CONF_FILE);
        return -1;
    }

    while (fgets(confline, sizeof(confline), conf))
    {
        if ('#' == confline[0] || '\r' == confline[0] || '\n' == confline[0]) /* comment */
            continue;
 
        sep = strchr(confline, ','); /* AABBCCDDEEFF,FFEEDDCCBBAA */
        if (!sep)
        {
            YC_DBG_PRINT("conf line \"%s\" format error.\r\n", confline);
            fclose(conf);
            return -2;
        }

        if (12 != (sep - confline)) /* AABBCCDDEEFF */
        {
            YC_DBG_PRINT("conf mac \"%s\" format error.\r\n", confline);
            fclose(conf);
            return -3;
        }

        i++;
    }

    YC_DBG_PRINT("conf count %d\r\n", i);
    g_wns_conf_size = i;

    g_wns_conf_table = malloc(i * sizeof(wns_conf_s));
    if (!g_wns_conf_table)
    {
        YC_DBG_PRINT("malloc conf table error.\r\n");
        fclose(conf);
        return -4;
    }

    i = 0;
    rewind(conf);
    while (fgets(confline, sizeof(confline), conf))
    {
        if ('#' == confline[0] || '\r' == confline[0] || '\n' == confline[0])
            continue;
 
        sep = strchr(confline, ',');

        memcpy(g_wns_conf_table[i].rmac, confline, 12);
        memcpy(g_wns_conf_table[i].cmac, sep + 1, 12);

        i++;
    }

    fclose(conf);

    return 0;
}

static int wns_epoll_add(int efd, int fd)
{
    int ret;
    struct epoll_event event;

    memset(&event, 0, sizeof(event));
    event.data.fd = fd;
    event.events = EPOLLIN; /* | EPOLLET */
    ret = epoll_ctl(efd, EPOLL_CTL_ADD, fd, &event);

    return ret;
}

static int wns_epoll_del(int efd, int fd)
{
    int ret;
    struct epoll_event event;

    memset(&event, 0, sizeof(event));
    event.data.fd = fd;
    event.events = EPOLLIN; /* | EPOLLET */
    ret = epoll_ctl(efd, EPOLL_CTL_DEL, fd, &event);

    return ret;
}

static void wns_close_fd(int efd, int fd)
{
    g_wns_socket_table[fd].state = -1;

    wns_epoll_del(efd, fd);
    close(fd);

    return;
}

static int wns_socket_handle(int fd, const uchar *buf, int len)
{
    int ret = -1;
    uchar *pos;

    if (0 == g_wns_socket_table[fd].state)
    {
        YC_DBGASSERT((len == 14) || (len == 32));
    
        pos = buf;
        while ()
        {
            switch (*pos++)
            {
                case WNS_TLV_TYPE_VER:
                {
                    YC_DBGASSERT(1 == *pos);
                    pos += *pos;
                    if (0 == *pos)
                    
                    break;
                }
                default:
                {
                    YC_DBG_PRINT("socket %d unknown tlv type %hhu\r\n", fd, *pos);
                    break;
                }
            }
        }
    }
    else if (1 == g_wns_socket_table[fd].state)
    {

    }
    else
    {

    }

    return ret;
}

static void *wns_work_thread(void *arg)
{
    int i;
    int count;
    int fd;
    uint len;
    int qfd;
    int efd;
    int ret;
    uchar *buf;

    struct epoll_event *events;

    qfd = (int)(ulong)arg;
    YC_DBGASSERT(qfd >= 0);

    WNS_DEBUG("thread entry %ld, qfd %d\r\n", pthread_self(), qfd);

    pthread_detach(pthread_self());

    efd = epoll_create(WNS_EPOLL_SIZE);
    if (efd < 0)
    {
        YC_DBG_PRINT("thread epoll error\r\n");
        close(qfd);
        return NULL;
    }

    ret = wns_epoll_add(efd, qfd);
    if (ret)
    {
        YC_DBG_PRINT("thread epoll add qfd error\r\n");
        close(qfd);
        close(efd);
        return NULL;
    }

    events = malloc(WNS_EPOLL_SIZE * sizeof(struct epoll_event));
    if (!events)
    {
        YC_DBG_PRINT("thread epoll malloc error\r\n");
        close(qfd);
        close(efd);
        return NULL;
    }

    buf = malloc(WNS_THREAD_RECV_BUF_SIZE);
    if (!buf)
    {
        YC_DBG_PRINT("thread buf malloc error\r\n");
        close(qfd);
        close(efd);
        free(events);
        return NULL;
    }

    while (1)
    {
        count = epoll_wait(efd, events, WNS_EPOLL_SIZE, -1);
        for (i = 0; i < count; i++)
        {
            if (events[i].events & EPOLLIN)
            {
                fd = events[i].data.fd;

                if (fd == qfd)
                {
                    ret = yc_queue_read(qfd, &fd, &len);
                    if (!ret)
                    {
                        YC_DBGASSERT(sizeof(fd) == len);

                        WNS_DEBUG("thread entry %ld, new fd %d\r\n", pthread_self(), fd);

                        if (-1 != g_wns_socket_table[fd].state)
                        {
                            YC_DBG_PRINT("thread entry %ld, fd %d have used.\r\n", pthread_self(), fd);
                            continue;
                        }
                        else
                        {
                            g_wns_socket_table[fd].state = 0;
                        }

                        ret = wns_epoll_add(efd, fd);
                        if (ret)
                        {
                            YC_DBG_PRINT("thread epoll add fd error\r\n");
                        }
                    }
                    else
                    {
                        YC_DBG_PRINT("thread read queue error\r\n");
                    }
                }
                else
                {
                    ret = recv(fd, buf, WNS_THREAD_RECV_BUF_SIZE, 0);
                    if (0 == ret)
                    {
                        WNS_DEBUG("thread socket %d closed\r\n", fd);
                        wns_close_fd(efd, fd);
                    }
                    else if (ret > 0)
                    {
                        WNS_DEBUG("thread socket recv %d bytes\r\n", ret);
                        ret = wns_socket_handle(fd, buf, ret);
                        if (ret < 0)
                        {
                            WNS_DEBUG("socket data error = %d\r\n", ret);
                            wns_close_fd(efd, fd);
                        }
                    }
                    else
                    {
                        if (errno == EINTR || errno == EWOULDBLOCK || errno == EAGAIN)
                        {
                            /* continue */
                        }
                        else
                        {
                            WNS_DEBUG("thread socket %d error, close\r\n", fd);
                            wns_close_fd(efd, fd);
                        }
                    }
                }
            }
        }
    }

    return NULL;
}

static int wns_thread_init(void)
{
    int i;
    int qfd;
    int ret;
    pthread_t tid;
    char qname[16];

    for (i = 0; i < WNS_THREAD_NUM; i++)
    {
        sprintf(qname, "wnsq-%d", i);
        qfd = yc_queue_create(qname, WNS_THREAD_QUEUE_SIZE);
        if (qfd < 0)
        {
            YC_DBG_PRINT("create queue %d error\r\n", i);
            return -1;
        }

        ret = pthread_create(&tid, NULL, wns_work_thread, (void *)(ulong)qfd);
        if (ret)
        {
            YC_DBG_PRINT("create thread %d error\r\n", i);
            return -2;
        }

        WNS_DEBUG("thread %ld, qfd %d\r\n", tid, qfd);

        g_wns_thread_queue[i] = qfd;
    }

    return 0;
}

static int wns_server_init(void)
{
    int fd;
    int ret;
    struct sockaddr_in addr;

    signal(SIGHUP, SIG_IGN);
    signal(SIGPIPE, SIG_IGN);

    fd = socket(AF_INET, SOCK_STREAM, 0);
    if (fd < 0)
    {
        YC_DBG_PRINT("create socket error\r\n");
        return -1;
    }

    ret = 1;
    ret = setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, 
                    (const void *)&ret, sizeof(ret));
    if (ret)
    {
        YC_DBG_PRINT("setsockopt reuse addr error\r\n");
        return -2;
    }

    memset(&addr, 0, sizeof(addr));
    addr.sin_family = AF_INET;
    addr.sin_port = htons(WNS_LOCAL_LISTEN_PORT);
    addr.sin_addr.s_addr = INADDR_ANY;

	ret = bind(fd, (struct sockaddr *)&addr, sizeof(addr));
	if (ret)
	{
        YC_DBG_PRINT("socket bind error\r\n");
        return -3;
    }

    ret = listen(fd, SOMAXCONN);
    if (ret)
    {
        YC_DBG_PRINT("socket listen error\r\n");
        return -4;
    }

    g_wns_server_fd = fd;

    return 0;
}

static void wns_server_loop(void)
{
    int ret;
    socklen_t alen;
    struct sockaddr_in addr;

    memset(&addr, 0, sizeof(addr));
    alen = sizeof(addr);
    ret = accept(g_wns_server_fd, (struct sockaddr *)&addr, &alen);
    if (ret >= 0)
    {
        WNS_DEBUG("client %d (%s:%hu) connected.\r\n", ret, inet_ntoa(addr.sin_addr), ntohs(addr.sin_port));

        ret = yc_queue_write(g_wns_thread_queue[ret % WNS_THREAD_NUM], &ret, sizeof(ret));
        if (ret)
        {
            YC_DBG_PRINT("write queue %d error\r\n", g_wns_thread_queue[ret % WNS_THREAD_NUM]);
        }
    }
    else
    {
        YC_DBG_PRINT("socket accept error\r\n");
    }

    return;
}

int main(int argc, char *argv[])
{
    int ret;

    ret = wns_load_conf();

    if (!ret)
        ret = wns_thread_init();

    if (!ret)
        ret = wns_server_init();

    if (!ret)
    {
        WNS_DEBUG("wns server running...\r\n");

        while (1)
        {
            wns_server_loop();
        }
    }

    return ret;
}

