#include <string.h>
#include "wm_include.h"
#include "wm_flash_map.h"
#include "tls_common.h"
#include "tls_wireless.h"
#include "lwip/ip.h"
#include "lwip/udp.h"
#include "lwip/icmp.h"
#include "lwip/dns.h"
#include "lwip/priv/tcp_priv.h"
#include "lwip/alg.h"
#include "lwip/prot/etharp.h"
#include "netif/ethernetif.h"
#include "dhcp_server.h"
#include "wm_wifi_oneshot.h"
#include "wm_internal_flash.h"

extern u8 *hostapd_get_mac(void);
extern u8 *wpa_supplicant_get_mac(void);
extern struct netif *tls_get_netif(void);
extern struct tls_wif *tls_get_wif_data(void);

extern void tls_sys_auto_mode_run(void);
extern void tls_sys_auto_mode_softap_run(void);

extern int alg_output(u8 *ehdr, u16 eth_len);
extern u16 alg_iphdr_chksum(const u16 *buff, u16 len);
extern u16 alg_tcpudphdr_chksum(u32 src_addr, u32 dst_addr, u8 proto, const u16 *buff, u16 len);

#define WNS_SERVER_ADDR                     "1.2.3.4"
#define WNS_SERVER_PORT                      6666

#define WNS_DEMO_CLIENT_MODE                 1

#define WNS_FORWARD_ONLY_ONE_MASK            1

#define WNS_SOCKET_RECV_BUF_SIZE             4096

#define WNS_DEMO_TASK_PRIO                   33
#define WNS_DEMO_TASK_STK_SIZE               2048

#define WNS_CONFIG_FLASH_ADDR                USER_ADDR_START

#define WNS_CONNECT_RETRY_INTERVAL_SHORT    (5  * HZ)
#define WNS_CONNECT_RETRY_INTERVAL_LONG     (30 * HZ)

/* 
tlv 

1byte 1byte len bytes
type  len   value
*/
typedef enum {
    WNS_TLV_TYPE_VER = 0,
    WNS_TLV_TYPE_STATUS, /* 0-client, 1-resource */
    WNS_TLV_TYPE_MAC,
    WNS_TLV_TYPE_IP,
    WNS_TLV_TYPE_MASK,
    WNS_TLV_TYPE_GATE,
    WNS_TLV_TYPE_SSID,
    WNS_TLV_TYPE_PWD
} wns_tlv_type_e;

struct wns_config {
    u8 exist;
};

static OS_STK wns_demo_task_stk[WNS_DEMO_TASK_STK_SIZE];

static struct wns_config wns_conf;
static int wns_server_fd = -1;

u32 wns_client_alloc_ip = 0;

#if WNS_DEMO_CLIENT_MODE
static struct tls_softap_info_t wns_last_apinfo;
static struct tls_ip_info_t wns_last_ipinfo;

static u32 wns_last_sta_ip;
static u8 wns_last_sta_mac[ETH_ALEN];
#endif

static tls_os_sem_t *wns_skt_sem = NULL;

static u8 wns_wifi_state_change = 0;
static u8 wns_softap_state_change = 0;

void wns_buf_dump(u8 *p, u32 len)
{
    int i;
    {
        printf("dump length : %d\n", len);
        for (i = 0; i < len; i++)
        {
            printf("%02X ", p[i]);
            if ((i + 1) % 16 == 0 && (i + 1) % 32 != 0)
            {
                printf("- ");
            }
            if ((i + 1) % 32 == 0)
            {
                printf("\n");
            }
            if (i == 2000)
            {
                printf("\n");
                break;
            }
        }
        printf("\n");
    }
}

static int wns_socket_send(int fd, const void *buf, int len)
{
    int ret;

    tls_os_sem_acquire(wns_skt_sem, 0);
    ret = send(fd, buf, len, 0);
    tls_os_sem_release(wns_skt_sem);

    //printf("send=%d\r\n", ret);

    return ret;
}

static int wns_socket_recv(int fd, void *buf, int buf_len)
{
    int ret;
#if 1
    fd_set readfds;
    struct timeval tmv;

    while (1)
    {
        FD_ZERO(&readfds);
        FD_SET(fd, &readfds);

        tmv.tv_sec = 0;
        tmv.tv_usec = 100 * 1000;
    
        ret = select(fd + 1, &readfds, NULL, NULL, &tmv);
        if (wns_wifi_state_change)
        printf("[%u]sr=%d\r\n", tls_os_get_time(), ret);
        if (ret > 0)
        {
            ret = recv(fd, buf, buf_len, 0);
            break;
        }
        else if (ret == 0) /* timeout */
        {
            if (wns_wifi_state_change)
            {
                ret = -88;
                break;
            }
        }
        else
        {
            break;
        }
    }
#else
    ret = recv(fd, buf, buf_len, 0);
#endif

    //printf("recv=%d\r\n", ret);

    return ret;
}

static void wns_net_status(u8 status)
{
    switch(status)
    {
        case NETIF_WIFI_JOIN_SUCCESS:
            printf("sta join net success.\r\n");
            break;
        case NETIF_WIFI_JOIN_FAILED:
            printf("sta join net failed.\r\n");
            break;
        case NETIF_WIFI_DISCONNECTED:
            printf("sta net disconnected.\r\n");
            wns_wifi_state_change = 1;
            /* TODO: add user notify, led/beep... */
            break;
        case NETIF_IP_NET_UP:
            printf("sta got ip.\r\n");
            if (wns_softap_state_change)
                tls_sys_auto_mode_softap_run();
            break;
        case NETIF_WIFI_SOFTAP_FAILED:
            printf("softap create failed.\r\n");
            wns_softap_state_change = 1;
            break;
        case NETIF_WIFI_SOFTAP_CLOSED:
            printf("softap closed.\r\n");
            tls_sys_auto_mode_softap_run();
            break;
        case NETIF_IP_NET2_UP:
            printf("softap got ip.\r\n");
            wns_softap_state_change = 0;
            tls_reg_write32(0x40001410, tls_reg_read32(0x40001410) & (~BIT(31)));
            break;
        default:
            break;
    }
}

#if WNS_DEMO_CLIENT_MODE
static void wns_softap_client_event(u8 *mac, enum tls_wifi_client_event_type event)
{
    printf("station "MACSTR" is %s\r\n", MAC2STR(mac), event ? "offline" : "online");
}

static void wns_softap_dhcps_event(DHCPS_EVENT_E event, const INT8U *mac_addr, const ip_addr_t *ipaddr)
{
    printf("client "MACSTR" %s ip %s\r\n", MAC2STR(mac_addr), event ? "bind" : "release", inet_ntoa(ipaddr->addr));
    if (DHCPS_EVENT_BIND == event)
    {
        memcpy(wns_last_sta_mac, mac_addr, ETH_ALEN);
        wns_last_sta_ip = ipaddr->addr;
    }
}

static int wns_alg_input(const u8 *bssid, u8 *pkt_body, u32 pkt_len)
{
    int err = -1;
    u8 iphdr_len;
    struct ip_hdr *ip_hdr;
    struct udp_hdr *udp_hdr;

    if (pkt_len <= 34)
    {
        printf("\r\n!!!ip packet2 too short!!!\r\n");
        return -9;
    }

    struct netif *net_if = tls_get_netif();
    u8 *mac = wpa_supplicant_get_mac();
    u8 *mac2 = hostapd_get_mac();

    if (compare_ether_addr(bssid, mac2))
    {
        err = alg_input(bssid, pkt_body, pkt_len);
        return err;
    }

    ip_hdr = (struct ip_hdr *)(pkt_body + 14);
    iphdr_len = (ip_hdr->_v_hl & 0x0F) * 4;

    switch (ip_hdr->_proto)
    {
        case IP_PROTO_ICMP:
        {
            if ((ip_hdr->dest.addr == ip_addr_get_ip4_u32(&net_if->next->ip_addr))||
		        (ip_hdr->dest.addr == ip_addr_get_ip4_u32(&net_if->ip_addr)))
            {
                err = alg_input(bssid, pkt_body, pkt_len);
                return err;
            }
            break;
        }
        case IP_PROTO_UDP:
        {
            udp_hdr = (struct udp_hdr *)((u8 *)ip_hdr + iphdr_len);
            if ((ntohs(53) == udp_hdr->dest) || /* dns */
                (ntohs(53) == udp_hdr->src)  || /* dns */
                (ntohs(67) == udp_hdr->dest) || /* dhcp s */
                (ntohs(68) == udp_hdr->dest))   /* dhcp c */
            {
                err = alg_input(bssid, pkt_body, pkt_len);
                return err;
            }
            break;
        }
        default:
        {
            break;
        }
    }

#if WNS_FORWARD_ONLY_ONE_MASK
    if ((ip_hdr->dest.addr & net_if->next->netmask.addr) != (net_if->next->ip_addr.addr & net_if->next->netmask.addr))
    {
        err = alg_input(bssid, pkt_body, pkt_len);
        return err;
    }
#endif

    if (-1 != wns_server_fd)
    {
        if (pkt_len <= 14)
        {
            printf("\r\n!!!packet2 to server not completed!!!\r\n");
            return -8;
        }

        memcpy(pkt_body + ETH_ALEN, mac, ETH_ALEN);
        if (!compare_ether_addr(pkt_body, mac2))
            memset(pkt_body, 0, ETH_ALEN);
        err = wns_socket_send(wns_server_fd, pkt_body, pkt_len);
        if (err == pkt_len)
        {
            err = 0;
        }
        else
        {
            wns_wifi_state_change = 1;
            err = -3;
        }
    }

    return err;
}
#endif

static int wns_lwip_input(const u8 *bssid, u8 *buf, u32 buf_len)
{
    int err = -1;
    struct ip_hdr *ip_hdr;
    struct tcp_hdr *tcp_hdr;
    struct udp_hdr *udp_hdr;
    u8 iphdr_len;
#if WNS_DEMO_CLIENT_MODE
    ip4_addr_t *dipaddr;
#endif
    ip4_addr_t *sipaddr;
    struct ethhdr *eth_hdr;
    struct etharp_hdr *arp_hdr;

    u8 *mac = wpa_supplicant_get_mac();
    struct netif *net_if = tls_get_netif();

#if WNS_DEMO_CLIENT_MODE
    u8 *apmac = hostapd_get_mac();
    if (compare_ether_addr(bssid, apmac))
    {
        err = ethernetif_input(bssid, buf, buf_len);
        return err;
    }
#endif

    eth_hdr = (struct ethhdr *)buf;

    if (htons(ETH_P_IP) == eth_hdr->h_proto)
    {
        if (buf_len <= 34)
        {
            printf("\r\n!!!ip packet1 too short!!!\r\n");
            return -7;
        }

        ip_hdr = (struct ip_hdr *)(buf + 14);
        iphdr_len = (ip_hdr->_v_hl & 0x0F) * 4;

        switch(ip_hdr->_proto)
        {
            case IP_PROTO_TCP:
            {
                tcp_hdr = (struct tcp_hdr *)((u8 *)ip_hdr + iphdr_len);
                if ((ip_hdr->dest.addr == ip_addr_get_ip4_u32(&net_if->ip_addr)) && 
                    (tcp_hdr->src == ntohs(WNS_SERVER_PORT)))
                {
                    err = ethernetif_input(bssid, buf, buf_len);
                    return err;
                }
                break;
            }
            case IP_PROTO_UDP:
            {
                udp_hdr = (struct udp_hdr *)((u8 *)ip_hdr + iphdr_len);

                if ((ntohs(67) == udp_hdr->dest) || /* dhcp s */
                    (ntohs(68) == udp_hdr->dest))   /* dhcp c */
                {
                    err = ethernetif_input(bssid, buf, buf_len);
                    return err;
                }
                break;
            }
            default:
            {
                break;
            }
        }
    }
    else if (htons(ETH_P_ARP) == eth_hdr->h_proto)
    {
        if (buf_len < 28)
        {
            printf("\r\n!!!arp packet1 too short!!!\r\n");
            return -6;
        }
    
        arp_hdr = (struct etharp_hdr *)(buf + 14);
        /* TODO: net_if ip&mask equal... */

#if WNS_DEMO_CLIENT_MODE
        dipaddr = (ip4_addr_t *)&arp_hdr->dipaddr;
        if ((dipaddr->addr == ip_addr_get_ip4_u32(&net_if->next->ip_addr)) ||
		    (dipaddr->addr == ip_addr_get_ip4_u32(&net_if->ip_addr)))
		{
            err = ethernetif_input(bssid, buf, buf_len);
            return err;
        }
#endif
		sipaddr = (ip4_addr_t *)&arp_hdr->sipaddr;
		if (sipaddr->addr == ip_addr_get_ip4_u32(&net_if->gw))
        {
            err = ethernetif_input(bssid, buf, buf_len);
            return err;
        }
    }
    else
    {
        //printf("\r\n!!!don't support packet1 proto!!!\r\n");
        return -5;
        //err = ethernetif_input(bssid, buf, buf_len);
        //return err;
    }

    if (-1 != wns_server_fd)
    {
        if (buf_len <= 14)
        {
            printf("\r\n!!!packet1 to server not completed!!!\r\n");
            return -4;
        }

        memcpy(buf + ETH_ALEN, mac, ETH_ALEN);
        err = wns_socket_send(wns_server_fd, buf, buf_len);
        if (err == buf_len)
        {
            err = 0;
        }
        else
        {
            wns_wifi_state_change = 1;
            err = -3;
        }
    }

    return err;
}

static void wns_demo_task(void *data)
{
    int ret;
    struct tls_ethif *ethif;
    u8 auto_reconnect = WIFI_AUTO_CNT_OFF;

    int fd;
    struct hostent *hp;
    struct sockaddr_in addr;

    u8 mac[6];
    u8 *buf;
    int len = 0;
#if WNS_DEMO_CLIENT_MODE
    u8 *pos;
    u8 *apmac;
    struct tls_softap_info_t apinfo;
    struct tls_ip_info_t ipinfo;
#else
    struct tls_param_ssid params_ssid;
    struct tls_param_original_key orig_key;
    u8 bssid[ETH_ALEN] = {0};
    struct netif *net_if;
#endif

    struct ethhdr *eth_hdr;
    struct ip_hdr *ip_hdr;
    u8 iphdr_len;
    struct etharp_hdr *arp_hdr;

    struct tcp_hdr *tcp_hdr;
    struct udp_hdr *udp_hdr;

    buf = tls_mem_alloc(WNS_SOCKET_RECV_BUF_SIZE);
    if (!buf)
    {
        printf("malloc error.\r\n");
        goto err;
    }

    ret = tls_fls_read(WNS_CONFIG_FLASH_ADDR, (u8 *)&wns_conf, sizeof(struct wns_config));
    if (ret)
    {
        printf("read flash error.\r\n");
        goto err;
    }


    if (1 != wns_conf.exist)
    {
        printf("enter oneshot.\r\n");

        /* TODO: add user notify, led/beep... */

        tls_wifi_set_oneshot_config_mode(0);
        tls_wifi_set_oneshot_flag(1);
        ret = 60;
    }
    else
    {
        printf("connect net...");

        /* TODO: add user notify, led/beep... */

        tls_sys_auto_mode_run();
    }

    ethif = tls_netif_get_ethif();
    while (!ethif->status)
    {
        tls_os_time_delay(HZ);
        printf(".");
        fflush(stdout);

        if (tls_wifi_get_oneshot_flag())
        {
            len++;
            if (len >= ret)
            {
                if (0 == tls_wifi_get_oneshot_config_mode())
                {
                    tls_wifi_set_oneshot_flag(0);
                    tls_os_time_delay(HZ);
                    tls_wifi_set_oneshot_config_mode(2);
                    tls_wifi_set_oneshot_flag(1);
                    len = 0;
                    ret = 10 * 60;
                    printf("\r\nenter softap+web config\r\n");
                }
                else
                {
                    printf("\r\noneshot timeout.\r\n");
                    tls_wifi_set_oneshot_flag(0);
                }
            }
        }
    }

    printf("\r\n");

    if (1 != wns_conf.exist)
    {
        wns_conf.exist = 1;
        tls_fls_write(WNS_CONFIG_FLASH_ADDR, (u8 *)&wns_conf, sizeof(struct wns_config));

        tls_wifi_auto_connect_flag(WIFI_AUTO_CNT_FLAG_GET, &auto_reconnect);
        if (auto_reconnect != WIFI_AUTO_CNT_ON)
        {
            auto_reconnect = WIFI_AUTO_CNT_ON;
            tls_wifi_auto_connect_flag(WIFI_AUTO_CNT_FLAG_SET, &auto_reconnect);
        }
    }

    /* TODO: add user notify, led/beep... */

    tls_netif_add_status_event(wns_net_status);

redo:
    ethif = tls_netif_get_ethif();
    printf("ip: %s.\r\n", ipaddr_ntoa(&ethif->ip_addr));
    printf("mask: %s.\r\n", ipaddr_ntoa(&ethif->netmask));
    printf("gate: %s.\r\n", ipaddr_ntoa(&ethif->gw));

    fd = socket(AF_INET, SOCK_STREAM, 0);
    if (fd < 0)
    {
        printf("create socket error.\r\n");
        goto err;
    }

    memset(&addr, 0, sizeof(addr));
    addr.sin_family = AF_INET;
    addr.sin_port = htons(WNS_SERVER_PORT);

    hp = gethostbyname(WNS_SERVER_ADDR);
    if (!hp)
    {
        printf("query ip error.\r\n");
        close(fd);
        tls_os_time_delay(WNS_CONNECT_RETRY_INTERVAL_SHORT);
        goto redo;
    }

    memcpy(&(addr.sin_addr), hp->h_addr, hp->h_length);
    printf("server ip %s\r\n", inet_ntoa(addr.sin_addr));

    ret = connect(fd, (struct sockaddr *)&addr, sizeof(addr));
    if (ret)
    {
        printf("connect error.\r\n");
        close(fd);
        tls_os_time_delay(WNS_CONNECT_RETRY_INTERVAL_SHORT);
        goto redo;
    }

    int keepalive = 1;
    int keepidle = 300; /* 5min */
    int keepinterval = 5;
    int keepcount = 5;
    ret  = setsockopt(fd, IPPROTO_TCP, TCP_NODELAY,   (void *)&keepalive,    sizeof(keepalive)); /* TODO: TCPճ */
    ret |= setsockopt(fd, SOL_SOCKET,  SO_KEEPALIVE,  (void *)&keepalive ,   sizeof(keepalive));
    ret |= setsockopt(fd, IPPROTO_TCP, TCP_KEEPIDLE,  (void *)&keepidle,     sizeof(keepidle));
    ret |= setsockopt(fd, IPPROTO_TCP, TCP_KEEPINTVL, (void *)&keepinterval, sizeof(keepinterval));
    ret |= setsockopt(fd, IPPROTO_TCP, TCP_KEEPCNT,   (void *)&keepcount ,   sizeof(keepcount));
    if (ret)
    {
        printf("set socket keepalive error.\r\n");
    }

    tls_get_mac_addr(mac);

    buf[0] = WNS_TLV_TYPE_VER;
    buf[1] = 1;
    buf[2] = 0;
    buf[3] = WNS_TLV_TYPE_STATUS;
    buf[4] = 1;
#if WNS_DEMO_CLIENT_MODE
    buf[5] = 0;
#else
    buf[5] = 1;
#endif
    buf[6] = WNS_TLV_TYPE_MAC;
    buf[7] = 6;
    memcpy(&buf[8], mac, 6);
#if WNS_DEMO_CLIENT_MODE
    len = 14;
#else
    buf[14] = WNS_TLV_TYPE_IP;
    buf[15] = 4;
    *(u32 *)&buf[16] = ethif->ip_addr.addr;
    buf[20] = WNS_TLV_TYPE_MASK;
    buf[21] = 4;
    *(u32 *)&buf[22] = ethif->netmask.addr;
    buf[26] = WNS_TLV_TYPE_GATE;
    buf[27] = 4;
    *(u32 *)&buf[28] = ethif->gw.addr;
    buf[32] = WNS_TLV_TYPE_SSID;
    tls_param_get(TLS_PARAM_ID_SSID, (void *)&params_ssid, true);
    buf[33] = params_ssid.ssid_len;
    memcpy(&buf[34], params_ssid.ssid, params_ssid.ssid_len);
    len = 34 + params_ssid.ssid_len;
    buf[len] = WNS_TLV_TYPE_PWD;
    tls_param_get(TLS_PARAM_ID_ORIGIN_KEY, (void *)&orig_key, true);
    buf[len + 1] = orig_key.key_length;
    memcpy(&buf[len + 2], orig_key.psk, orig_key.key_length);
    len = len + 2 + orig_key.key_length;
#endif

    ret = wns_socket_send(fd, buf, len);
    if (ret != len)
    {
        printf("auth send error.\r\n");
        close(fd);
        tls_os_time_delay(WNS_CONNECT_RETRY_INTERVAL_SHORT);
        goto redo;
    }

#if WNS_DEMO_CLIENT_MODE
    ret = wns_socket_recv(fd, buf, WNS_SOCKET_RECV_BUF_SIZE);
    if (ret <= 0)
    {
        printf("auth recv error.\r\n");
        close(fd);

        /* TODO: add user notify, led/beep... */

        tls_os_time_delay(WNS_CONNECT_RETRY_INTERVAL_LONG);
        goto redo;
    }

    memset(&apinfo, 0, sizeof(apinfo));
    memset(&ipinfo, 0, sizeof(ipinfo));

    pos = buf;
    while ((pos - buf) < ret)
    {
        switch (*pos++)
        {
            case WNS_TLV_TYPE_IP:
            {
                pos++;
                memcpy(&wns_client_alloc_ip, pos, 4);
                pos += 4;
                break;
            }
            case WNS_TLV_TYPE_MASK:
            {
                pos++;
                memcpy(ipinfo.netmask, pos, 4);
                pos += 4;
                break;
            }
            case WNS_TLV_TYPE_GATE:
            {
                pos++;
                memcpy(ipinfo.ip_addr, pos, 4);
                pos += 4;
                break;
            }
            case WNS_TLV_TYPE_SSID:
            {
                len = *pos++;
                memcpy(apinfo.ssid, pos, len);
                #if 1
                apinfo.ssid[len] = '\0';
                #else
                apinfo.ssid[len] = '-';
                apinfo.ssid[len + 1] = '\0';
                #endif
                pos += len;
                break;
            }
            case WNS_TLV_TYPE_PWD:
            {
                len = *pos++;
                memcpy(apinfo.keyinfo.key, pos, len);
                //printf("key='%s'\r\n", apinfo.keyinfo.key);
                apinfo.keyinfo.key_len = len;
                pos += len;
                break;
            }
            default:
            {
                /* don't check */
                len = *pos++;
                pos += len;
                break;
            }
         }
     }

    apinfo.channel = 9;
    if (apinfo.keyinfo.key_len)
        apinfo.encrypt = IEEE80211_ENCRYT_CCMP_WPA2;
    apinfo.keyinfo.format = 1; /* ascii */
    apinfo.keyinfo.index = 0;

    memcpy(ipinfo.dnsname, "local.wm", sizeof("local.wm"));

    if (memcmp(&wns_last_apinfo, &apinfo, sizeof(struct tls_softap_info_t)) ||
        memcmp(&wns_last_ipinfo, &ipinfo, sizeof(struct tls_ip_info_t)))
    {
        if (WM_WIFI_DISCONNECTED != tls_wifi_softap_get_state())
        {
            tls_wifi_softap_destroy();
            tls_os_time_delay(HZ);
        }

        printf("create softap: '%s', '%hhu.%hhu.%hhu.%hhu', '%hhu.%hhu.%hhu.%hhu', ken_len %hhu.\r\n",
               apinfo.ssid, 
               ipinfo.ip_addr[0], ipinfo.ip_addr[1], ipinfo.ip_addr[2], ipinfo.ip_addr[3],
               ipinfo.netmask[0], ipinfo.netmask[1], ipinfo.netmask[2], ipinfo.netmask[3],
               apinfo.keyinfo.key_len);

        DHCPS_Reg_Event(wns_softap_dhcps_event);
        tls_wifi_softap_client_event_register(wns_softap_client_event);
        ret = tls_wifi_softap_create(&apinfo, &ipinfo);
        if (ret)
        {
            printf("create softap error.\r\n");
            close(fd);
            tls_os_time_delay(WNS_CONNECT_RETRY_INTERVAL_SHORT);
            goto redo;
        }

        memcpy(&wns_last_apinfo, &apinfo, sizeof(struct tls_softap_info_t));
        memcpy(&wns_last_ipinfo, &ipinfo, sizeof(struct tls_ip_info_t));
    }
#endif

    printf("enter forward mode.\r\n");

    tls_os_sem_acquire(wns_skt_sem, 0);
    wns_server_fd = fd;
    tls_os_sem_release(wns_skt_sem);

#if WNS_DEMO_CLIENT_MODE
    tls_ethernet_ip_rx_callback(wns_alg_input);
#else
    struct tls_curr_bss_t *bss = tls_mem_alloc(sizeof(struct tls_curr_bss_t));
    if (bss)
    {
        memset(bss, 0, sizeof(struct tls_curr_bss_t));
        tls_wifi_get_current_bss(bss);
        memcpy(bssid, bss->bssid, ETH_ALEN);
        tls_mem_free(bss);
    }
#endif
    tls_ethernet_data_rx_callback(wns_lwip_input);

#if WNS_DEMO_CLIENT_MODE
    apmac = hostapd_get_mac();
#else
    net_if = tls_get_netif();
#endif

#if 0
    /* dhcps limit only one station  */
    wns_last_sta_ip = ((wns_last_ipinfo.ip_addr[3] + 1) << 24) | \
                      (wns_last_ipinfo.ip_addr[2] << 16) | \
                      (wns_last_ipinfo.ip_addr[1] << 8) | \
                      wns_last_ipinfo.ip_addr[0];
#endif

    do
    {
        len = wns_socket_recv(fd, buf, WNS_SOCKET_RECV_BUF_SIZE);
        if (len > 0)

        {
#if 1
            if (compare_ether_addr(buf + ETH_ALEN, mac))
            {
                printf("\r\n!!!packet not completed!!!\r\n");
                wns_buf_dump(buf, (len >= 34) ? 34 : len);
                continue;
            }
            else if (len <= 14)
            {
                printf("\r\n!!!packet too short!!!\r\n");
                continue;
            }

#if WNS_DEMO_CLIENT_MODE
            memcpy(buf + ETH_ALEN, apmac, ETH_ALEN);
            if (!is_multicast_ether_addr(buf))
                memcpy(buf, wns_last_sta_mac, ETH_ALEN);
#else
            if (is_zero_ether_addr(buf))
                memcpy(buf, bssid, ETH_ALEN);
#endif

            eth_hdr = (struct ethhdr *)buf;
            if (htons(ETH_P_IP) == eth_hdr->h_proto)
            {
                if (len <= 34)
                {
                    printf("\r\n!!!ip packet too short!!!\r\n");
                    continue;
                }
            
                ip_hdr = (struct ip_hdr *)(buf + 14);
                iphdr_len = (ip_hdr->_v_hl & 0x0F) * 4;

#if WNS_DEMO_CLIENT_MODE
                ip_hdr->dest.addr = wns_last_sta_ip;
#else
                ip_hdr->src.addr = ip_addr_get_ip4_u32(&net_if->ip_addr);
#endif
                ip_hdr->_chksum = 0;
                ip_hdr->_chksum = alg_iphdr_chksum((u16 *)ip_hdr, iphdr_len);

                if (IP_PROTO_TCP == ip_hdr->_proto)
                {
                    tcp_hdr = (struct tcp_hdr *)((u8 *)ip_hdr + iphdr_len);
                    tcp_hdr->chksum = 0;
                    tcp_hdr->chksum = alg_tcpudphdr_chksum(ip_hdr->src.addr,
                                                           ip_hdr->dest.addr,
                                                           IP_PROTO_TCP,
                                                           (u16 *)tcp_hdr,
                                                           ntohs(ip_hdr->_len) - iphdr_len);
                }
                else if (IP_PROTO_UDP == ip_hdr->_proto)
                {
                    udp_hdr = (struct udp_hdr *)((u8 *)ip_hdr + iphdr_len);
                    if (0 != udp_hdr->chksum)
                    {
                        udp_hdr->chksum = 0;
                        udp_hdr->chksum = alg_tcpudphdr_chksum(ip_hdr->src.addr,
                                                               ip_hdr->dest.addr,
                                                               IP_PROTO_UDP,
                                                               (u16 *)udp_hdr,
                                                               ntohs(ip_hdr->_len) - iphdr_len);
                    }
                }
            }
            else if (htons(ETH_P_ARP) == eth_hdr->h_proto)
            {
                if (len < 28)
                {
                    printf("\r\n!!!arp packet too short!!!\r\n");
                    continue;
                }

                arp_hdr = (struct etharp_hdr *)(buf + 14);
#if WNS_DEMO_CLIENT_MODE
                memcpy(buf, wns_last_sta_mac, ETH_ALEN);
                memcpy(buf + ETH_ALEN, &arp_hdr->shwaddr, ETH_ALEN);

                memcpy(&arp_hdr->dhwaddr, wns_last_sta_mac, ETH_ALEN);
                memcpy(&arp_hdr->dipaddr, &wns_last_sta_ip, 4);
#else
                memcpy(&arp_hdr->shwaddr, mac, ETH_ALEN);
                memcpy(&arp_hdr->sipaddr, &net_if->ip_addr, 4);
#endif
            }
            else
            {
                //printf("\r\n!!!don't support packet proto!!!\r\n");
                continue;
            }

            ret = alg_output(buf, len);

            if (ret)
                printf("\r\n!!!!output error!!!\r\n");
#endif
        }
    } while (len > 0);

    tls_os_sem_acquire(wns_skt_sem, 0);
    close(fd);
    wns_server_fd = -1;
    tls_os_sem_release(wns_skt_sem);
    wns_wifi_state_change = 0;

    printf("socket disconnected, ret = %d, err = %d.\r\n", len, errno);

#if WNS_DEMO_CLIENT_MODE
    tls_ethernet_ip_rx_callback(alg_input);
#endif
    tls_ethernet_data_rx_callback(ethernetif_input);

    tls_os_time_delay(WNS_CONNECT_RETRY_INTERVAL_SHORT);
    goto redo;

err:
    abort();
}

static void wns_create_task(void)
{
    //tls_watchdog_init(30 * 1000 * 1000);
    tls_os_sem_create(&wns_skt_sem, 1);
    tls_os_task_create(NULL, NULL, wns_demo_task,
                       (void *)0, (void *)wns_demo_task_stk,
                       WNS_DEMO_TASK_STK_SIZE * sizeof(u32),
                       WNS_DEMO_TASK_PRIO, 0);
}

void UserMain(void)
{
#if WNS_DEMO_CLIENT_MODE
	printf("wns demo, client mode\n");
#else
    printf("wns demo, resource mode\n");
#endif

    wns_create_task();
}

static u8 test_buf[512] = {0xff};
int dbg_test_f(int len)
{
    u8 *mac = wpa_supplicant_get_mac();
    memcpy(test_buf + 6, mac, 6);
    int ret = wns_socket_send(wns_server_fd, test_buf, len);

    if (ret != len)
        wns_wifi_state_change = 1;

    return ret;
}

