/*
 * This file is part of the MicroPython project, http://micropython.org/
 *
 * The MIT License (MIT)
 *
 * Copyright (c) 2013-2017 Damien P. George
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "wm_include.h"

#include "oofatfs/ff.h"
#include "oofatfs/diskio.h"

#include "wm_vfs_fat.h"
#include "wm_sd_driver.h"

#define VFS_FAT_FS_BLOCK_SIZE         (4096)
#define VFS_FAT_FS_SECTOR_SIZE        (FF_MAX_SS)

#define WM_VFS_FAT_DBG printf

static BYTE wm_sd_inited = 0;
static fs_user_mount_t *wm_sd_vfs_fat = NULL;

static int wm_vfs_fat_read (BYTE *buff, UINT sector, UINT count)
{
    DRESULT res;
	int result;

	if (count > 1)
		result = SD_ReadMultiBlock(sector, buff, count);
	else
		result = SD_ReadSingleBlock(sector, buff);

    if (result)
		res = RES_ERROR;
	else
		res = RES_OK;

	return res;
}

static int wm_vfs_fat_write (const BYTE *buff, UINT sector, UINT count)
{
    DRESULT res;
	int result;

	if (count > 1)
		result = SD_WriteMultiBlock(sector, buff, count);
	else
		result = SD_WriteSingleBlock(sector, buff);

	// translate the reslut code here
	if (result)
		res = RES_ERROR;
	else
		res = RES_OK;

	return res;
}

static int wm_vfs_fat_ioctl(BYTE cmd, void *buff)
{
	//DRESULT res;
	//int result;

	switch (cmd) {
	        case CTRL_SYNC:
                return RES_OK;
			case GET_BLOCK_SIZE:
				*(DWORD *)buff = VFS_FAT_FS_BLOCK_SIZE;
				return RES_OK;
			case GET_SECTOR_SIZE:
				*(DWORD *)buff = VFS_FAT_FS_SECTOR_SIZE;
				return RES_OK;
			case GET_SECTOR_COUNT:
				*(DWORD *)buff = SD_GetCapacity() / VFS_FAT_FS_SECTOR_SIZE;
				return RES_OK;
			case MMC_GET_CSD:
				SD_GetCSD(buff);
				return RES_OK;
			case MMC_GET_CID:
				SD_GetCID(buff);
				return RES_OK;
			case MMC_GET_TYPE:
				*(UINT *)buff = SD_GetCardType();
				return RES_OK;
		    case IOCTL_INIT:
		    case IOCTL_STATUS:
                if (wm_sd_inited)
                {
                    *(BYTE *)buff = 0;
		            return RES_OK;
		        }
		        else
		        {
                    *(BYTE *)buff = STA_NOINIT;
		            return RES_NOTRDY;
		        }
		}

    //printf("vfs_fat ioctl, cmd = %d\r\n", cmd);
	return RES_PARERR;
}

int wm_vfs_fat_init(void)
{
    // Initialise the local flash filesystem.
    if (SD_Init()) {
        WM_VFS_FAT_DBG("failed to init sdcard\r\n");
        return -1;
    }
    wm_sd_inited = 1;

    // init the vfs object
    wm_sd_vfs_fat = tls_mem_alloc(sizeof(*wm_sd_vfs_fat));
    if (!wm_sd_vfs_fat) {
        WM_VFS_FAT_DBG("failed to init filesystem\r\n");
        return -2;
    }

    fs_user_mount_t *vfs_fat = wm_sd_vfs_fat;
    vfs_fat->flags = 0;

    vfs_fat->flags |= FSUSER_NATIVE | FSUSER_HAVE_IOCTL;
    vfs_fat->fatfs.drv = vfs_fat;
    vfs_fat->readblocks = wm_vfs_fat_read; // native version
    vfs_fat->writeblocks = wm_vfs_fat_write; // native version
    vfs_fat->ioctl = wm_vfs_fat_ioctl;

    // Create it if needed, and mount in on /flash.
    FRESULT res = f_mount(&vfs_fat->fatfs);
    if (res == FR_NO_FILESYSTEM) {
        // no filesystem, so create a fresh one
        WM_VFS_FAT_DBG("create filesystem\r\n");
        uint8_t working_buf[FF_MAX_SS];
        res = f_mkfs(&vfs_fat->fatfs, FM_FAT, 0, working_buf, sizeof(working_buf));
        if (res == FR_OK) {
            // success creating fresh LFS
        } else {
            WM_VFS_FAT_DBG("failed to create filesystem, err = %d\r\n", res);
            return -3;
        }

        // set label
        //f_setlabel(&vfs_fat->fatfs, "wm_fs");
    } else if (res == FR_OK) {
        // mount sucessful

    } else {
    //fail:
        WM_VFS_FAT_DBG("failed to mount flash, err = %d\r\n", res);
        return -4;
    }

    res = f_chdir(&vfs_fat->fatfs, "/");
    //printf("res = %d\r\n", res);

    return 0;
}

fs_user_mount_t * wm_vfs_fat_get_ctx(void)
{
    return wm_sd_vfs_fat;
}

