#include "wm_include.h"
#include "netif/ethernetif.h"
#include "utils.h"
#include "task.h"

#include "oofatfs/ff.h"
#include "vfs/wm_vfs_fat.h"


#define TCPDUMP_DEFAULT_NANE        ("/w60xdump/wm_lwip.pcap")

#define TCPDUMP_MAX_MSG             (10)
#define PCAP_FILE_HEADER_SIZE       (24)
#define PCAP_PKTHDR_SIZE            (16)

#define PCAP_FILE_ID                (0xA1B2C3D4)
#define PCAP_VERSION_MAJOR          (0x200)
#define PCAP_VERSION_MINOR          (0x400)
#define GREENWICH_MEAN_TIME         (0)
#define PRECISION_OF_TIME_STAMP     (0)
#define MAX_LENTH_OF_CAPTURE_PKG    (0xFFFF)

#define LINKTYPE_NULL               (0)
#define LINKTYPE_ETHERNET           (1)                 /* also for 100Mb and up */
#define LINKTYPE_EXP_ETHERNET       (2)                 /* 3Mb experimental Ethernet */
#define LINKTYPE_AX25               (3)
#define LINKTYPE_PRONET             (4)
#define LINKTYPE_CHAOS              (5)
#define LINKTYPE_TOKEN_RING         (6)                 /* DLT_IEEE802 is used for Token Ring */
#define LINKTYPE_ARCNET             (7)
#define LINKTYPE_SLIP               (8)
#define LINKTYPE_PPP                (9)
#define LINKTYPE_FDDI               (10)
#define LINKTYPE_PPP_HDLC           (50)                /* PPP in HDLC-like framing */
#define LINKTYPE_PPP_ETHER          (51)                /* NetBSD PPP-over-Ethernet */
#define LINKTYPE_ATM_RFC1483        (100)               /* LLC/SNAP-encapsulated ATM */
#define LINKTYPE_RAW                (101)               /* raw IP */
#define LINKTYPE_SLIP_BSDOS         (102)               /* BSD/OS SLIP BPF header */
#define LINKTYPE_PPP_BSDOS          (103)               /* BSD/OS PPP BPF header */
#define LINKTYPE_C_HDLC             (104)               /* Cisco HDLC */
#define LINKTYPE_IEEE802_11         (105)               /* IEEE 802.11 (wireless) */
#define LINKTYPE_ATM_CLIP           (106)               /* Linux Classical IP over ATM */
#define LINKTYPE_LOOP               (108)               /* OpenBSD loopback */
#define LINKTYPE_LINUX_SLL          (113)               /* Linux cooked socket capture */
#define LINKTYPE_LTALK              (114)               /* Apple LocalTalk hardware */
#define LINKTYPE_ECONET             (115)               /* Acorn Econet */
#define LINKTYPE_CISCO_IOS          (118)               /* For Cisco-internal use */
#define LINKTYPE_PRISM_HEADER       (119)               /* 802.11+Prism II monitor mode */
#define LINKTYPE_AIRONET_HEADER     (120)               /* FreeBSD Aironet driver stuff */


#define TCPDUMP_DBG printf

#define         TCPDUMP_QUEUE_SIZE            32
static tls_os_queue_t *tcpump_msgq = NULL;

#define         TCPDUMP_TASK_PRIO             35
#define         TCPDUMP_TASK_STK_SIZE        256
static OS_STK   tcpdump_task_stk[TCPDUMP_TASK_STK_SIZE];

static struct netif *tcpdump_if = NULL;

static FIL fp;
static FIL *pfp = NULL;

#define PACP_FILE_HEADER_CREATE(_head)                          \
do {                                                            \
    (_head)->magic = PCAP_FILE_ID;                              \
    (_head)->version_major = PCAP_VERSION_MAJOR;                \
    (_head)->version_minor = PCAP_VERSION_MINOR;                \
    (_head)->thiszone = GREENWICH_MEAN_TIME;                    \
    (_head)->sigfigs = PRECISION_OF_TIME_STAMP;                 \
    (_head)->snaplen = MAX_LENTH_OF_CAPTURE_PKG;                \
    (_head)->linktype = LINKTYPE_ETHERNET;                      \
} while (0)

#define PACP_PKTHDR_CREATE(_head, _p)                           \
do{                                                             \
    (_head)->ts.tv_sec = tls_os_get_time() / HZ;    \
    (_head)->ts.tv_msec = tls_os_get_time() % HZ;   \
    (_head)->caplen = _p->tot_len;                              \
    (_head)->len = _p->tot_len;                                 \
} while (0)

struct wm_pcap_file_header
{
    u32 magic;
    u16 version_major;
    u16 version_minor;
    s32 thiszone;
    u32 sigfigs;
    u32 snaplen;
    u32 linktype;
};

struct wm_timeval
{
    u32 tv_sec;
    u32 tv_msec;
};

struct wm_pcap_pkthdr
{
    struct wm_timeval ts;
    u32 caplen;
    u32 len;
};


static netif_linkoutput_fn link_output;
static netif_input_fn input;

static const char *name;
static char *filename;

static int (*tcpdump_write)(const void *buf, int len);

/* set file name */
static void wm_tcpdump_filename_set(const char *name)
{
    filename = strdup(name);
}

/* delete file name */
static void wm_tcpdump_filename_del(void)
{
    name = NULL;
    if (filename != NULL)
        tls_mem_free(filename);

    filename = NULL;
}

/* get tx data */
static err_t _netif_linkoutput(struct netif *netif, struct pbuf *p)
{
    if (p != NULL)
    {
        pbuf_ref(p);

        if (tls_os_queue_send(tcpump_msgq, (void *)p, 0) != TLS_OS_SUCCESS)
        {
            pbuf_free(p);
        }
    }
    return link_output(netif, p);
}

/* get rx data */
static err_t _netif_input(struct pbuf *p, struct netif *inp)
{
    if (p != NULL)
    {
        pbuf_ref(p);

        if (tls_os_queue_send(tcpump_msgq, (void *)p, 0) != TLS_OS_SUCCESS)
        {
            pbuf_free(p);
        }
    }
    return input(p, inp);
}

/* import pcap file into your PC through file-system */
static int wm_tcpdump_pcap_file_write(const void *buf, int len)
{
    int length;
    FRESULT res;

    if (filename == NULL)
    {
        TCPDUMP_DBG("file name is null!\n");
        return -1;
    }

    if ((len == 0) && (pfp != NULL))
    {
        TCPDUMP_DBG("ip mess error and close file!\n");
        //f_close(pfp);
        //pfp = NULL;
        return -2;
    }

    if (pfp == NULL)
    {
        fs_user_mount_t *vfs = wm_vfs_fat_get_ctx();

        res = f_open (&vfs->fatfs, &fp, (const TCHAR*)filename, FA_WRITE | FA_CREATE_ALWAYS);
        if (res != FR_OK)
        {
            TCPDUMP_DBG("open file failed!\n");
            return -3;
        }
        pfp = &fp;
    }

    res = f_write (pfp, buf, len, (UINT*)&length);
    if (length != len)
    {
        TCPDUMP_DBG("write data failed, length: %d\n", length);
        f_close(pfp);
        pfp = NULL;
        return -4;
    }

    return 0;
}

/* write ip mess and print */
static void wm_tcpdump_ip_mess_write(struct pbuf *p)
{
    u8 *buf = (u8 *)tls_mem_alloc(p->tot_len);

    if (!buf)
    {
        TCPDUMP_DBG("mem err, len = %d!\n", p->tot_len);
        return;
    }

    pbuf_copy_partial(p, buf, p->tot_len, 0);

    /* write ip mess */
    if (tcpdump_write != NULL)
        tcpdump_write(buf, p->tot_len);

    tls_mem_free(buf);
}

/* write pcap file header */
static int wm_tcpdump_pcap_file_init(void)
{
    struct wm_pcap_file_header file_header;
    int res = 0;

    /* in rdb mode does not need to write pcap file header */
    if ((tcpdump_write != NULL) && (tcpdump_write == wm_tcpdump_pcap_file_write))
    {
        PACP_FILE_HEADER_CREATE(&file_header);
        res = tcpdump_write(&file_header, sizeof(file_header));
    }

    return res;
}

static void tcpdump_task(void *data)
{
    void *msg;
    int ret;
    struct pbuf *pbuf = NULL;
    struct wm_pcap_pkthdr pkthdr;

    for( ; ; )
	{
		ret = tls_os_queue_receive(tcpump_msgq, (void **)&msg, 0, 0);

		if (TLS_OS_SUCCESS == ret)
		{
            pbuf = (struct pbuf *)msg;

            /* write pkthdr */
            if ((tcpdump_write != NULL) && (tcpdump_write == wm_tcpdump_pcap_file_write))
            {
                PACP_PKTHDR_CREATE(&pkthdr, pbuf);
                tcpdump_write(&pkthdr, sizeof(pkthdr));
            }

            wm_tcpdump_ip_mess_write(pbuf);
            pbuf_free(pbuf);
            pbuf = NULL;
		}
		/* tcpdump deinit, the mailbox does not receive the data, exits the thread*/
        else
        {
            TCPDUMP_DBG("tcpdump error!\n");

            f_close(pfp);
            pfp = NULL;

            tcpdump_write = NULL;
            wm_tcpdump_filename_del();

            /* ɾԼ */
            vTaskDelete(NULL);
            return;
        }
	}
}

int wm_tcpdump_init(enum tls_wifi_op_mode mode)
{
    struct netif *netif = tls_get_netif();

    if (STATION_MODE == mode)
    {
        tcpdump_if = netif;
    }
    else if (SOFTAP_MODE == mode)
    {
        tcpdump_if = netif->next;
    }
    else
    {
        return -1;
    }

    tls_os_queue_create(&tcpump_msgq, TCPDUMP_QUEUE_SIZE);

    tls_os_task_create(NULL, NULL, tcpdump_task,
                       (void *)0, (void *)tcpdump_task_stk,
                       TCPDUMP_TASK_STK_SIZE * sizeof(u32),
                       TCPDUMP_TASK_PRIO, 0);

    tcpdump_write = wm_tcpdump_pcap_file_write;
    name = TCPDUMP_DEFAULT_NANE;
    wm_tcpdump_filename_set(name);

    /* linkoutput and input init */
    u32 cpu_sr = tls_os_set_critical();
    link_output = tcpdump_if->linkoutput;
    tcpdump_if->linkoutput = _netif_linkoutput;

    input = tcpdump_if->input;
    tcpdump_if->input = _netif_input;
    tls_os_release_critical(cpu_sr);
    /* linkoutput and input init */

    /* write pcap file header */
    wm_tcpdump_pcap_file_init();

    TCPDUMP_DBG("tcpdump start!\n");

    return 0;
}

void wm_tcpdump_deinit(void)
{
    /* linkoutput and input deinit */
    u32 cpu_sr = tls_os_set_critical();
    tcpdump_if->linkoutput = link_output;
    tcpdump_if->input = input;
    tcpdump_if = NULL;
    tls_os_release_critical(cpu_sr);
    /* linkoutput and input deinit */

    tls_os_queue_delete(tcpump_msgq);
    tcpump_msgq = NULL;

    TCPDUMP_DBG("tcpdump stop!\n");
}

