#include <stdio.h>
#include <stddef.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <pthread.h>
#include <fcntl.h>
#include <stdarg.h>
#include <sys/types.h>
#include <sys/select.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <arpa/inet.h>

#include "../include/yc_opt.h"
#if !YC_WINDOWS_LITE
#include <netinet/in.h>
#include <linux/tcp.h>
#endif

#include "../include/yc_basetype.h"
#include "../include/yc_queue.h"
#include "../include/yc_socket.h"
#include "../include/list.h"
#include "../include/yc_debug.h"


/* ==============================================
   socket server̻߳ƴ
   ʱ -lpthread ѡ
================================================= */


#define IP_ADDR_MAX_LEN                          16
#define RCV_BUF_MAX_LEN                          2048
#define RCV_LEN_MAX                              1600
#define SKT_QUEUE_NAME_LEN                       64

/* ========================== unix ׽ =========================== */
#define YC_UNIX_NAME_MAX_LEN                     128

/* ǰ̵ */
#define YC_UNIX_LOCAL_PROC_NAME                 "/proc/self/exe"

/* ========================== unix ׽ =========================== */


typedef enum tagSocket_msg_type{
    YC_SOCKET_SERVER_CLOSE_MSG = 1,
    YC_SOCKET_CLIENT_CLOSE_MSG = 2,
    YC_SOCKET_CLOSE_CLIENT_MSG = 3
}yc_socket_msg_e;

typedef struct tagSocket_msg{
    yc_socket_msg_e msg;
    int iSockfd;
}yc_socket_msg_s;

typedef struct tagSocket_client{
    struct list_head stNode;
    int iAcceptQuefd; /* accept̶߳ */
    int iSockfd;
    int iQuefd;
    void *pfRcvCallback;
    void *pArg;
    pthread_t stThrdID;
}yc_socket_client_s;

struct tagSocket_server_ctx{
    struct list_head stNode;
    pthread_rwlock_t stClientLock;

    int iQuefd;
    pthread_t stThreadID;
    void *pfRcvCallback;
    void *pArg;
};

static void yc_socket_set_keepalive(int iSockFd)
{
#if !YC_WINDOWS_LITE
    int ret;
    int keepalive = 1; /*  */
    int keepidle = 1800; /* ڶ೤ʱûκ̽ */
    int keepinterval = 5; /* ̽ʱʱ */
    int keepcount = 5; /* ̽ⳢԴ */

    ret = setsockopt(iSockFd, SOL_SOCKET, SO_KEEPALIVE, (void *)&keepalive , sizeof(keepalive));
    if (0 != ret)
    {
        YC_DBG_PRINT("[%d] failed to set keepalive.\r\n", iSockFd);
    }
    ret = setsockopt(iSockFd, IPPROTO_TCP, TCP_KEEPIDLE, (void*)&keepidle , sizeof(keepidle));
    if (0 != ret)
    {
        YC_DBG_PRINT("[%d] failed to set keepidle.\r\n", iSockFd);
    }
    ret = setsockopt(iSockFd, IPPROTO_TCP, TCP_KEEPINTVL, (void *)&keepinterval , sizeof(keepinterval));
    if (0 != ret)
    {
        YC_DBG_PRINT("[%d] failed to set keepintvl.\r\n", iSockFd);
    }
    ret = setsockopt(iSockFd, IPPROTO_TCP, TCP_KEEPCNT, (void *)&keepcount , sizeof(keepcount));
    if (0 != ret)
    {
        YC_DBG_PRINT("[%d] failed to set keepcnt.\r\n", iSockFd);
    }
#endif

    return;
}

static int yc_socket_check_connect(int sock, int connect_res, uint wait_time)
{
	struct timeval tmvTimeout = {wait_time / 1000, wait_time * 1000};
	int result;
	fd_set fdSet;
	socklen_t len;
	int valopt = 0;

	FD_ZERO(&fdSet);
	FD_SET(sock, &fdSet);

	if (connect_res < 0)
	{
		//if (EINPROGRESS == errno)
    	{
			result = select(sock + 1, NULL, &fdSet, NULL, &tmvTimeout);
			if (result < 0)
			{
				return YC_ERROR_FAILED;
			}
			else if (result == 0)
			{
				/* timeout */
				return YC_ERROR_TIMEOUT;
			}
			else
			{

				len = sizeof(valopt);
				if (getsockopt(sock, SOL_SOCKET, SO_ERROR, (void *) &valopt, &len) < 0)
				{
					/* failed to read delayed error */
					return YC_ERROR_FAILED;
				}
				else if (valopt)
				{
					/* delayed error = valopt */
					return YC_ERROR_FAILED;
				}
			}
        }
	}

	return YC_ERROR_SUCCESS;
}

/* ֵ>0Ϊյݳȣ<=0ʧ */
static int __yc_socket_recv(int sockfd, void *buf, size_t len, int flags,
                            struct sockaddr *dest_addr, socklen_t *addrlen)
{
    int ret;

redo:
    if ((NULL == dest_addr) || (NULL == addrlen))
    {
        ret = recv(sockfd, buf, len, 0);
    }
    else
    {
        ret = recvfrom(sockfd, buf, len, 0, dest_addr, addrlen);
    }

    if (ret > 0)
    {
        /* successfully */
    }
    else if (ret == 0)
    {
        ret = YC_ERROR_FAILED;
    }
    else
    {
        if ((errno == EINTR) || (errno == EWOULDBLOCK) || (errno == EAGAIN))
        {
            goto redo;
        }
        else
        {
            ret = YC_ERROR_FAILED;
        }
    }

    return ret;
}

/* ֵֻгɹʧ */
static int __yc_socket_send(int sockfd, const void *buf, size_t len, int flags,
                            const struct sockaddr *dest_addr, socklen_t addrlen)
{
    int iRet = 0;
    uint uiWriteLen = 0;

resnd:
    while (uiWriteLen != len)
    {
        if ((NULL != dest_addr) && (0 != addrlen))
            iRet = sendto(sockfd, buf + uiWriteLen, len - uiWriteLen, flags, dest_addr, addrlen);
        else
            iRet = send(sockfd, buf + uiWriteLen, len - uiWriteLen, flags);

        if (iRet > 0)
        {
            uiWriteLen += iRet;
        }
        else if ((iRet < 0) && ((errno == EINTR) ||
                                (errno == EWOULDBLOCK) ||
                                (errno == EAGAIN)))
        {
            goto resnd;
        }
        else
        {
            iRet = YC_ERROR_FAILED;
            break;
        }
    }

    if (uiWriteLen == len)
        iRet = YC_ERROR_SUCCESS;

    return iRet;
}

/* ֵ>0Ϊյݳȣ<=0ʧ */
static int __yc_select_recv(int sockfd, uint time, void *buf, uint len, int flags,
                            struct sockaddr *addr, socklen_t *addrlen)
{
    int ret;
    fd_set readset;
    struct timeval  tv;
    struct timeval *ptv = NULL;

    if (0 != time)
    {
        memset(&tv, 0, sizeof(struct timeval));
        tv.tv_sec  = time / 1000;
        tv.tv_usec = time % 1000;
        ptv = &tv;
    }

    FD_ZERO(&readset);
    FD_SET(sockfd, &readset);

    ret = select(sockfd + 1, &readset, NULL, NULL, ptv);
    if (ret > 0)
    {
        if (FD_ISSET(sockfd, &readset))
        {
             ret = __yc_socket_recv(sockfd, buf, len, flags, addr, addrlen);

             FD_CLR(sockfd, &readset);
        }
        else
        {
            ret = YC_ERROR_FAILED;
        }
    }
    else if (ret == 0)
    {
        ret = YC_ERROR_TIMEOUT;
    }
    else
    {
        ret = YC_ERROR_FAILED;
    }

    return ret;
}

static void *udp_server_rcv_thread(IN void *pData)
{
    int iRet;
    socklen_t iAddrLen;
    char szSrcIP[IP_ADDR_MAX_LEN];
    struct sockaddr_in stAddr;
    char acBuf[RCV_BUF_MAX_LEN];
    yc_socket_server_s *server_handle = NULL;
    yc_udp_server_rcv_pf pfCallback = NULL;
    fd_set readset;
    int maxfd;
    yc_socket_msg_s stMsg;
    uint msglen = 0;

    YC_DBGASSERT(NULL != pData);

    server_handle = (yc_socket_server_s *)pData;

    for ( ; ; )
	{
        FD_ZERO(&readset);
        FD_SET(server_handle->iSockfd, &readset);
        FD_SET(server_handle->ctx->iQuefd, &readset);
        maxfd = YC_MAX(server_handle->iSockfd, server_handle->ctx->iQuefd);

        iRet = select(maxfd + 1, &readset, NULL, NULL, NULL);
        if (iRet > 0)
        {
            if (FD_ISSET(server_handle->ctx->iQuefd, &readset))
            {
                memset(&stMsg, 0, sizeof(stMsg));
                iRet = yc_queue_read(server_handle->ctx->iQuefd, &stMsg, &msglen);
                if (YC_ERROR_SUCCESS == iRet)
                {
                    YC_DBGASSERT(sizeof(stMsg) == msglen);
                    YC_DBGASSERT(YC_SOCKET_SERVER_CLOSE_MSG == stMsg.msg);

                    //pthread_exit(NULL);
                    break;
                }
                else
                {
                    yc_debug_print(YC_DEBUG_TYPE_ERROR, "read que failed");
                }
                FD_CLR(server_handle->ctx->iQuefd, &readset);
            }

            if (FD_ISSET(server_handle->iSockfd, &readset))
            {
                memset(&stAddr, 0, sizeof(stAddr));
		        iAddrLen = sizeof(stAddr);
		        memset(acBuf, 0, RCV_BUF_MAX_LEN);
                iRet = __yc_socket_recv(server_handle->iSockfd, acBuf, RCV_LEN_MAX,
                                        0, (struct sockaddr *)&stAddr, &iAddrLen);
                if (iRet > 0)
                {
                    szSrcIP[0] = '\0';
        		    strcpy(szSrcIP, inet_ntoa(stAddr.sin_addr));
        		    pfCallback = (yc_udp_server_rcv_pf)server_handle->ctx->pfRcvCallback;
        			pfCallback(server_handle->iSockfd, server_handle->ctx->pArg,
        			           acBuf, iRet, szSrcIP, ntohs(stAddr.sin_port));
                }
                FD_CLR(server_handle->iSockfd, &readset);
            }
        }
	}

    return NULL;
}

static void tcp_server_close_client(yc_socket_server_ctx_s *ctx, int iSockfd)
{
    int iRet;
    yc_socket_msg_s stMsg;
    yc_socket_client_s *client = NULL;
    yc_socket_client_s *client_next = NULL;

    memset(&stMsg, 0, sizeof(stMsg));
    stMsg.msg = YC_SOCKET_SERVER_CLOSE_MSG;

    pthread_rwlock_wrlock(&ctx->stClientLock);
    list_for_each_entry_safe(client, client_next, &ctx->stNode, stNode)
    {
        if (iSockfd == client->iSockfd)
        {
            iRet = yc_queue_write(client->iQuefd, &stMsg, sizeof(stMsg));
            if (YC_ERROR_SUCCESS != iRet)
            {
                yc_debug_print(YC_DEBUG_TYPE_ERROR, "client %d queue %d write failed.", client->iSockfd, client->iQuefd);
            }

            list_del(&client->stNode);
            break;
        }
    }
    pthread_rwlock_unlock(&ctx->stClientLock);

    iRet = pthread_join(client->stThrdID, NULL);
    if (0 != iRet)
    {
        yc_debug_print(YC_DEBUG_TYPE_ERROR, "client %d join failed.", client->iSockfd);
    }

    yc_queue_destroy(client->iQuefd);
    close(client->iSockfd);
    yc_free(client);

    return;
}

static void tcp_server_client_close(yc_socket_server_ctx_s *ctx, int iSockfd)
{
    int iRet;
    pthread_t stThrdID;
    yc_socket_client_s *client = NULL;
    yc_socket_client_s *client_next = NULL;

    pthread_rwlock_wrlock(&ctx->stClientLock);
    list_for_each_entry_safe(client, client_next, &ctx->stNode, stNode)
    {
        if (iSockfd == client->iSockfd)
        {
            memcpy(&stThrdID, &client->stThrdID, sizeof(pthread_t));
            list_del(&client->stNode);
            yc_queue_destroy(client->iQuefd);
            close(client->iSockfd);
            yc_free(client);
            break;
        }
    }
    pthread_rwlock_unlock(&ctx->stClientLock);

    iRet = pthread_join(stThrdID, NULL);
    if (0 != iRet)
    {
        yc_debug_print(YC_DEBUG_TYPE_ERROR, "client %d join failed.", iSockfd);
    }

    return;
}

static void tcp_server_close_all_client(yc_socket_server_ctx_s *ctx)
{
    int iRet;
    yc_socket_msg_s stMsg;
    yc_socket_client_s *client = NULL;
    yc_socket_client_s *client_next = NULL;

    memset(&stMsg, 0, sizeof(stMsg));
    stMsg.msg = YC_SOCKET_SERVER_CLOSE_MSG;

    pthread_rwlock_rdlock(&ctx->stClientLock);
    list_for_each_entry(client, &ctx->stNode, stNode)
    {
        iRet = yc_queue_write(client->iQuefd, &stMsg, sizeof(stMsg));
        if (YC_ERROR_SUCCESS != iRet)
        {
            yc_debug_print(YC_DEBUG_TYPE_ERROR, "client %d queue %d write failed.", client->iSockfd, client->iQuefd);
        }
    }
    pthread_rwlock_unlock(&ctx->stClientLock);

    pthread_rwlock_wrlock(&ctx->stClientLock);
    list_for_each_entry_safe(client, client_next, &ctx->stNode, stNode)
    {
        iRet = pthread_join(client->stThrdID, NULL);
        if (0 != iRet)
        {
            yc_debug_print(YC_DEBUG_TYPE_ERROR, "client %d join failed.", client->iSockfd);
        }

        list_del(&client->stNode);
        yc_queue_destroy(client->iQuefd);
        close(client->iSockfd);
        yc_free(client);
    }
    pthread_rwlock_unlock(&ctx->stClientLock);

    return;
}

static void *tcp_server_rcv_thread(IN void *pData)
{
    int iRet;
    char szSrcIP[IP_ADDR_MAX_LEN];
    socklen_t iAddrLen;
    struct sockaddr_in stAddr;
    char acBuf[RCV_BUF_MAX_LEN];
    yc_socket_client_s *client = NULL;
    tcp_server_rcv_pf pfCallback = NULL;
    fd_set readset;
    int maxfd;
    yc_socket_msg_s stMsg;
    uint msglen = 0;

    YC_DBGASSERT(NULL != pData);

    client = (yc_socket_client_s *)pData;
    pfCallback = (tcp_server_rcv_pf)client->pfRcvCallback;

    memset(&stAddr, 0, sizeof(stAddr));
    iAddrLen = sizeof(stAddr);
    szSrcIP[0] = '\0';

    if (0 == getpeername(client->iSockfd, (struct sockaddr *)&stAddr, &iAddrLen))
        strcpy(szSrcIP, inet_ntoa(stAddr.sin_addr));

    pfCallback(client->iSockfd, client->pArg, NULL, YC_SOCKET_CLIENT_CONNECTED, szSrcIP, ntohs(stAddr.sin_port));

    for ( ; ; )
	{
        FD_ZERO(&readset);
        FD_SET(client->iSockfd, &readset);
        FD_SET(client->iQuefd, &readset);
        maxfd = YC_MAX(client->iSockfd, client->iQuefd);

        iRet = select(maxfd + 1, &readset, NULL, NULL, NULL);
        if (iRet > 0)
        {
            if (FD_ISSET(client->iQuefd, &readset))
            {
                memset(&stMsg, 0, sizeof(yc_socket_msg_s));
                iRet = yc_queue_read(client->iQuefd, &stMsg, &msglen);
                if (YC_ERROR_SUCCESS == iRet)
                {
                    YC_DBGASSERT(sizeof(stMsg) == msglen);
                    YC_DBGASSERT(YC_SOCKET_SERVER_CLOSE_MSG == stMsg.msg);

                    //pthread_exit(NULL);
                    break;
                }
                else
                {
                    yc_debug_print(YC_DEBUG_TYPE_ERROR, "read que failed");
                }
                FD_CLR(client->iQuefd, &readset);
            }

            if (FD_ISSET(client->iSockfd, &readset))
            {
 		        memset(acBuf, 0, RCV_BUF_MAX_LEN);

                iRet = __yc_socket_recv(client->iSockfd, acBuf, RCV_LEN_MAX, 0, NULL, 0);
                if (iRet > 0)
                {
         			pfCallback(client->iSockfd, client->pArg, acBuf, iRet, szSrcIP, ntohs(stAddr.sin_port));
                }
                else
                {
                    pfCallback(client->iSockfd, client->pArg, NULL, YC_SOCKET_CLIENT_CLOSED, szSrcIP, ntohs(stAddr.sin_port));

                    memset(&stMsg, 0, sizeof(yc_socket_msg_s));
                    stMsg.iSockfd = client->iSockfd;
                    stMsg.msg = YC_SOCKET_CLIENT_CLOSE_MSG;

                    iRet = yc_queue_write(client->iAcceptQuefd, &stMsg, sizeof(stMsg));
                    if (YC_ERROR_SUCCESS != iRet)
                    {
                        yc_debug_print(YC_DEBUG_TYPE_ERROR, "client %d error, write que failed.", client->iSockfd);
                    }
                    break;
                }
                FD_CLR(client->iSockfd, &readset);
            }
        }
	}

    return NULL;
}

static void *tcp_server_accept_thread(IN void *pData)
{
    int iRet;
    socklen_t iAddrLen;
    struct sockaddr_in stAddr;
    yc_socket_server_s *server = NULL;
    fd_set readset;
    int maxfd;
    yc_socket_msg_s stMsg;
    uint msglen = 0;
    int iClientfd;
    int iQuefd;
    char acQueName[SKT_QUEUE_NAME_LEN];
    yc_socket_client_s *client = NULL;

    YC_DBGASSERT(NULL != pData);

    server = (yc_socket_server_s *)pData;

    for ( ; ; )
	{
        FD_ZERO(&readset);
        FD_SET(server->iSockfd, &readset);
        FD_SET(server->ctx->iQuefd, &readset);
        maxfd = YC_MAX(server->iSockfd, server->ctx->iQuefd);

        iRet = select(maxfd + 1, &readset, NULL, NULL, NULL);
        if (iRet > 0)
        {
            if (FD_ISSET(server->ctx->iQuefd, &readset))
            {
                memset(&stMsg, 0, sizeof(stMsg));
                iRet = yc_queue_read(server->ctx->iQuefd, &stMsg, &msglen);
                if (YC_ERROR_SUCCESS == iRet)
                {
                    YC_DBGASSERT(sizeof(stMsg) == msglen);

                    if (YC_SOCKET_SERVER_CLOSE_MSG == stMsg.msg)
                    {
                        tcp_server_close_all_client(server->ctx);

                        //pthread_exit(NULL);
                        break;
                    }
                    else if (YC_SOCKET_CLIENT_CLOSE_MSG == stMsg.msg)
                    {
                        tcp_server_client_close(server->ctx, stMsg.iSockfd);
                    }
                    else if (YC_SOCKET_CLOSE_CLIENT_MSG == stMsg.msg)
                    {
                        tcp_server_close_client(server->ctx, stMsg.iSockfd);
                    }
                }
                else
                {
                    yc_debug_print(YC_DEBUG_TYPE_ERROR, "read que %d failed.", server->ctx->iQuefd);
                }
                FD_CLR(server->ctx->iQuefd, &readset);
            }

            if (FD_ISSET(server->iSockfd, &readset))
            {
                memset(&stAddr, 0, sizeof(stAddr));
		        iAddrLen = sizeof(stAddr);
                iClientfd = accept(server->iSockfd, (struct sockaddr *)&stAddr, &iAddrLen);
                if (iClientfd > -1)
                {
                    client = yc_malloc(sizeof(yc_socket_client_s));
                    if (NULL != client)
                    {
                        acQueName[0] = '\0';
                        sprintf(acQueName, "que-skt-%d", iClientfd);
                        iQuefd = yc_queue_create(acQueName, 3);
                        if (iQuefd >= 0)
                        {
                            memset(client, 0, sizeof(yc_socket_client_s));
                            client->iAcceptQuefd = server->ctx->iQuefd;
                            client->iQuefd = iQuefd;
                            client->iSockfd = iClientfd;
                            client->pfRcvCallback = server->ctx->pfRcvCallback;
                            client->pArg = server->ctx->pArg;
                            INIT_LIST_HEAD(&client->stNode);
                            yc_socket_set_keepalive(iClientfd);

                            iRet = pthread_create(&client->stThrdID, NULL, tcp_server_rcv_thread, client);
                            if (0 == iRet)
                            {
                                pthread_rwlock_wrlock(&server->ctx->stClientLock);
                                list_add(&client->stNode, &server->ctx->stNode);
                                pthread_rwlock_unlock(&server->ctx->stClientLock);
                            }
                            else
                            {
                                yc_queue_destroy(iQuefd);
                                yc_free(client);
                                close(iClientfd);
                            }
                        }
                        else
                        {
                            yc_free(client);
                            close(iClientfd);
                        }
                    }
                }
                FD_CLR(server->iSockfd, &readset);
            }
        }
	}

    return NULL;
}

static int unix_get_owner_name(OUT char *szOwnerName)
{
    int iRet = YC_ERROR_FAILED;
    char *pcPos = NULL;
    unsigned int uiPos = 0;
    char szOwnerPath[RCV_BUF_MAX_LEN];

    szOwnerPath[0] = '\0';
    iRet = readlink(YC_UNIX_LOCAL_PROC_NAME, szOwnerPath, RCV_BUF_MAX_LEN);
    if(-1 != iRet)
    {
        pcPos = szOwnerPath;/* õȫ·Ҫ */
        uiPos = strlen(szOwnerPath);
        while('/' != pcPos[uiPos - 1])
        {
            uiPos--;
        }
        strcpy(szOwnerName, pcPos + uiPos);
        iRet = YC_ERROR_SUCCESS;
    }
    else
    {
        iRet = YC_ERROR_FAILED;
    }

    return iRet;
}

static void *unix_dgram_server_rcv_thread(IN void *pData)
{
    int iRet;
    socklen_t iAddrLen;
    char szSrcName[YC_UNIX_NAME_MAX_LEN];
    struct sockaddr_un stAddr;
    char acBuf[RCV_BUF_MAX_LEN];
    yc_socket_server_s *server_handle = NULL;
    yc_unix_server_rcv_pf pfCallback = NULL;
    fd_set readset;
    int maxfd;
    yc_socket_msg_s stMsg;
    uint msglen = 0;

    YC_DBGASSERT(NULL != pData);

    server_handle = (yc_socket_server_s *)pData;

    for ( ; ; )
	{
        FD_ZERO(&readset);
        FD_SET(server_handle->iSockfd, &readset);
        FD_SET(server_handle->ctx->iQuefd, &readset);
        maxfd = YC_MAX(server_handle->iSockfd, server_handle->ctx->iQuefd);

        iRet = select(maxfd + 1, &readset, NULL, NULL, NULL);
        if (iRet > 0)
        {
            if (FD_ISSET(server_handle->ctx->iQuefd, &readset))
            {
                memset(&stMsg, 0, sizeof(stMsg));
                iRet = yc_queue_read(server_handle->ctx->iQuefd, &stMsg, &msglen);
                if (YC_ERROR_SUCCESS == iRet)
                {
                    YC_DBGASSERT(sizeof(stMsg) == msglen);
                    YC_DBGASSERT(YC_SOCKET_SERVER_CLOSE_MSG == stMsg.msg);

                    //pthread_exit(NULL);
                    break;
                }
                else
                {
                    yc_debug_print(YC_DEBUG_TYPE_ERROR, "read que failed");
                }
                FD_CLR(server_handle->ctx->iQuefd, &readset);
            }

            if (FD_ISSET(server_handle->iSockfd, &readset))
            {
                memset(&stAddr, 0, sizeof(stAddr));
		        iAddrLen = sizeof(stAddr);
		        memset(acBuf, 0, RCV_BUF_MAX_LEN);
                iRet = __yc_socket_recv(server_handle->iSockfd, acBuf, RCV_LEN_MAX,
                                        0, (struct sockaddr *)&stAddr, &iAddrLen);
                if (iRet > 0)
                {
                    iAddrLen -= offsetof(struct sockaddr_un, sun_path);
        		    stAddr.sun_path[iAddrLen] = '\0';
        		    szSrcName[0] = '\0';
        		    sscanf(stAddr.sun_path, YC_UNIX_DGRAM_REMOTE_NAME, szSrcName);
        		    szSrcName[strlen(szSrcName) - strlen(".socket")] = '\0';
        		    pfCallback = (yc_unix_server_rcv_pf)server_handle->ctx->pfRcvCallback;
        			pfCallback(server_handle->iSockfd, szSrcName, server_handle->ctx->pArg, acBuf, iRet);
                }
                FD_CLR(server_handle->iSockfd, &readset);
            }
        }
	}

    return NULL;
}

static void *unix_stream_server_rcv_thread(IN void *pData)
{
    int iRet;
    char szSrcName[YC_UNIX_NAME_MAX_LEN];
    socklen_t iAddrLen;
    struct sockaddr_un stAddr;
    char acBuf[RCV_BUF_MAX_LEN];
    yc_socket_client_s *client = NULL;
    yc_unix_server_rcv_pf pfCallback = NULL;
    fd_set readset;
    int maxfd;
    yc_socket_msg_s stMsg;
    uint msglen = 0;

    YC_DBGASSERT(NULL != pData);

    client = (yc_socket_client_s *)pData;
    pfCallback = (yc_unix_server_rcv_pf)client->pfRcvCallback;

    memset(&stAddr, 0, sizeof(stAddr));
    iAddrLen = sizeof(stAddr);
    szSrcName[0] = '\0';

    if (0 == getpeername(client->iSockfd, (struct sockaddr *)&stAddr, &iAddrLen))
    {
        iAddrLen -= offsetof(struct sockaddr_un, sun_path);
	    stAddr.sun_path[iAddrLen] = '\0';
	    sscanf(stAddr.sun_path, YC_UNIX_DGRAM_REMOTE_NAME, szSrcName);
	    szSrcName[strlen(szSrcName) - strlen(".socket")] = '\0';
    }

    pfCallback(client->iSockfd, szSrcName, client->pArg, NULL, YC_SOCKET_CLIENT_CONNECTED);

    for ( ; ; )
	{
        FD_ZERO(&readset);
        FD_SET(client->iSockfd, &readset);
        FD_SET(client->iQuefd, &readset);
        maxfd = YC_MAX(client->iSockfd, client->iQuefd);

        iRet = select(maxfd + 1, &readset, NULL, NULL, NULL);
        if (iRet > 0)
        {
            if (FD_ISSET(client->iQuefd, &readset))
            {
                memset(&stMsg, 0, sizeof(yc_socket_msg_s));
                iRet = yc_queue_read(client->iQuefd, &stMsg, &msglen);
                if (YC_ERROR_SUCCESS == iRet)
                {
                    YC_DBGASSERT(sizeof(stMsg) == msglen);
                    YC_DBGASSERT(YC_SOCKET_SERVER_CLOSE_MSG == stMsg.msg);

                    //pthread_exit(NULL);
                    break;
                }
                else
                {
                    yc_debug_print(YC_DEBUG_TYPE_ERROR, "read que failed");
                }
                FD_CLR(client->iQuefd, &readset);
            }

            if (FD_ISSET(client->iSockfd, &readset))
            {
		        memset(acBuf, 0, RCV_BUF_MAX_LEN);

                iRet = __yc_socket_recv(client->iSockfd, acBuf, RCV_LEN_MAX, 0, NULL, 0);
                if (iRet > 0)
                {
        			pfCallback(client->iSockfd, szSrcName, client->pArg, acBuf, iRet);
                }
                else
                {
                    pfCallback(client->iSockfd, szSrcName, client->pArg, NULL, YC_SOCKET_CLIENT_CLOSED);

                    memset(&stMsg, 0, sizeof(yc_socket_msg_s));
                    stMsg.iSockfd = client->iSockfd;
                    stMsg.msg = YC_SOCKET_CLIENT_CLOSE_MSG;

                    iRet = yc_queue_write(client->iAcceptQuefd, &stMsg, sizeof(stMsg));
                    if (YC_ERROR_SUCCESS != iRet)
                    {
                        yc_debug_print(YC_DEBUG_TYPE_ERROR, "client %d error, write que failed.", client->iSockfd);
                    }
                    break;
                }
                FD_CLR(client->iSockfd, &readset);
            }
        }
	}

    return NULL;
}

static void *unix_stream_server_accept_thread(IN void *pData)
{
    int iRet;
    socklen_t iAddrLen;
    struct sockaddr_un stAddr;
    yc_socket_server_s *server = NULL;
    fd_set readset;
    int maxfd;
    yc_socket_msg_s stMsg;
    uint msglen = 0;
    int iClientfd;
    int iQuefd;
    char acQueName[SKT_QUEUE_NAME_LEN];
    yc_socket_client_s *client = NULL;

    YC_DBGASSERT(NULL != pData);

    server = (yc_socket_server_s *)pData;

    for ( ; ; )
	{
        FD_ZERO(&readset);
        FD_SET(server->iSockfd, &readset);
        FD_SET(server->ctx->iQuefd, &readset);
        maxfd = YC_MAX(server->iSockfd, server->ctx->iQuefd);

        iRet = select(maxfd + 1, &readset, NULL, NULL, NULL);
        if (iRet > 0)
        {
            if (FD_ISSET(server->ctx->iQuefd, &readset))
            {
                memset(&stMsg, 0, sizeof(stMsg));
                iRet = yc_queue_read(server->ctx->iQuefd, &stMsg, &msglen);
                if (YC_ERROR_SUCCESS == iRet)
                {
                    YC_DBGASSERT(sizeof(stMsg) == msglen);

                    if (YC_SOCKET_SERVER_CLOSE_MSG == stMsg.msg)
                    {
                        tcp_server_close_all_client(server->ctx);

                        //pthread_exit(NULL);
                        break;
                    }
                    else if (YC_SOCKET_CLIENT_CLOSE_MSG == stMsg.msg)
                    {
                        tcp_server_client_close(server->ctx, stMsg.iSockfd);
                    }
                    else if (YC_SOCKET_CLOSE_CLIENT_MSG == stMsg.msg)
                    {
                        tcp_server_close_client(server->ctx, stMsg.iSockfd);
                    }
                }
                else
                {
                    yc_debug_print(YC_DEBUG_TYPE_ERROR, "read que %d failed.", server->ctx->iQuefd);
                }
                FD_CLR(server->ctx->iQuefd, &readset);
            }

            if (FD_ISSET(server->iSockfd, &readset))
            {
                memset(&stAddr, 0, sizeof(stAddr));
		        iAddrLen = sizeof(stAddr);
                iClientfd = accept(server->iSockfd, (struct sockaddr *)&stAddr, &iAddrLen);
                if (iClientfd > -1)
                {
                    client = yc_malloc(sizeof(yc_socket_client_s));
                    if (NULL != client)
                    {
                        acQueName[0] = '\0';
                        sprintf(acQueName, "que-skt-%d", iClientfd);
                        iQuefd = yc_queue_create(acQueName, 3);
                        if (iQuefd >= 0)
                        {
                            memset(client, 0, sizeof(yc_socket_client_s));
                            client->iAcceptQuefd = server->ctx->iQuefd;
                            client->iQuefd = iQuefd;
                            client->iSockfd = iClientfd;
                            client->pfRcvCallback = server->ctx->pfRcvCallback;
                            client->pArg = server->ctx->pArg;
                            INIT_LIST_HEAD(&client->stNode);

                            iRet = pthread_create(&client->stThrdID, NULL, unix_stream_server_rcv_thread, client);
                            if (0 == iRet)
                            {
                                pthread_rwlock_wrlock(&server->ctx->stClientLock);
                                list_add(&client->stNode, &server->ctx->stNode);
                                pthread_rwlock_unlock(&server->ctx->stClientLock);
                            }
                            else
                            {
                                yc_queue_destroy(iQuefd);
                                yc_free(client);
                                close(iClientfd);
                            }
                        }
                        else
                        {
                            yc_free(client);
                            close(iClientfd);
                        }
                    }
                }
                FD_CLR(server->iSockfd, &readset);
            }
        }
	}

    return NULL;
}

int yc_udp_send(IN const char *pcDstIP, IN ushort usDstPort,
                IN const void *pData,   IN uint   uiDataLen)
{
    int iRet = YC_ERROR_FAILED;
    int iSocketFd = -1;
    struct sockaddr_in stAddr;

    iSocketFd = socket(AF_INET, SOCK_DGRAM, 0);
    if (-1 == iSocketFd)
    {
        return YC_ERROR_FAILED;
    }

    memset(&stAddr, 0, sizeof(stAddr));
    stAddr.sin_family = AF_INET;
    stAddr.sin_port = htons(usDstPort);
    stAddr.sin_addr.s_addr = inet_addr(pcDstIP);

    iRet = __yc_socket_send(iSocketFd, pData, uiDataLen,
                            0, (struct sockaddr *)&stAddr, sizeof(stAddr));

    close(iSocketFd);

    return iRet;
}

int yc_tcp_send(IN const char *pcDstIP, IN ushort usDstPort,
                IN const void *pData,   IN uint uiDataLen)
{
    int iRet = YC_ERROR_FAILED;
    int iSocketFd = -1;
    struct sockaddr_in stAddr;

    iSocketFd = socket(AF_INET, SOCK_STREAM, 0);
    if (-1 == iSocketFd)
    {
        return YC_ERROR_FAILED;
    }

    memset(&stAddr, 0, sizeof(stAddr));
    stAddr.sin_family = AF_INET;
    stAddr.sin_port = htons(usDstPort);
    stAddr.sin_addr.s_addr = inet_addr(pcDstIP);
    iRet = connect(iSocketFd, (struct sockaddr *)&stAddr, sizeof(stAddr));
    if (0 != iRet)
    {
        close(iSocketFd);
        return YC_ERROR_FAILED;
    }

    iRet = __yc_socket_send(iSocketFd, pData, uiDataLen, 0, NULL, 0);

    close(iSocketFd);

    return iRet;
}

int yc_udp_send_and_rcv(IN uint uiTimeOut,
                        IN const char *pcDstIP, IN ushort usDstPort,
                        IN const void *pInData, IN uint uiDataLen,
                        OUT void *pOutData,     INOUT uint *puiOutDataLen)
{
    int iRet = YC_ERROR_FAILED;
    int iSocketFd = -1;
    struct sockaddr_in stAddr;

    if (NULL == puiOutDataLen)
        return YC_ERROR_PARAM;

    iSocketFd = socket(AF_INET, SOCK_DGRAM, 0);
    if (-1 == iSocketFd)
    {
        return YC_ERROR_FAILED;
    }

    memset(&stAddr, 0, sizeof(stAddr));
    stAddr.sin_family = AF_INET;
    stAddr.sin_port = htons(usDstPort);
    stAddr.sin_addr.s_addr = inet_addr(pcDstIP);

    iRet = __yc_socket_send(iSocketFd, pInData, uiDataLen, 0,
                            (struct sockaddr *)&stAddr, sizeof(stAddr));

    if (YC_ERROR_SUCCESS == iRet)
    {
        iRet = __yc_select_recv(iSocketFd, uiTimeOut, pOutData, *puiOutDataLen,
                                0, NULL, NULL);
        if (iRet > 0)
        {
            *puiOutDataLen = iRet;

            iRet = YC_ERROR_SUCCESS;
        }
    }

    close(iSocketFd);

    return iRet;
}

int yc_tcp_send_and_rcv(IN uint uiTimeOut,
                        IN const char *pcDstIP, IN ushort usDstPort,
                        IN const void *pInData, IN uint uiDataLen,
                        OUT void *pOutData,     INOUT uint *puiOutDataLen)
{
    int iRet = YC_ERROR_FAILED;
    int iSocketFd = -1;
    struct sockaddr_in stAddr;

    if (NULL == puiOutDataLen)
        return YC_ERROR_PARAM;

    iSocketFd = socket(AF_INET, SOCK_STREAM, 0);
    if (-1 == iSocketFd)
    {
        return YC_ERROR_FAILED;
    }

    memset(&stAddr, 0, sizeof(stAddr));
    stAddr.sin_family = AF_INET;
    stAddr.sin_port = htons(usDstPort);
    stAddr.sin_addr.s_addr = inet_addr(pcDstIP);
    iRet = connect(iSocketFd, (struct sockaddr *)&stAddr, sizeof(stAddr));
    if (0 != iRet)
    {
        close(iSocketFd);
        return YC_ERROR_FAILED;
    }

    iRet = __yc_socket_send(iSocketFd, pInData, uiDataLen, 0, NULL, 0);
    if (YC_ERROR_SUCCESS == iRet)
    {
        iRet = __yc_select_recv(iSocketFd, uiTimeOut, pOutData, *puiOutDataLen,
                                0, NULL, NULL);
        if (iRet > 0)
        {
            *puiOutDataLen = iRet;

            iRet = YC_ERROR_SUCCESS;
        }
    }

    close(iSocketFd);

    return iRet;
}

int yc_socket_send(IN int iSocketID,
                   IN const void *pData,   IN uint uiDataLen,
                   IN const char *pcDstIP, IN ushort usDstPort)
{
    int iRet = YC_ERROR_FAILED;
    socklen_t addrlen = 0;
    struct sockaddr_in  stAddr;
    struct sockaddr_in *pstAddr = NULL;

    if ((NULL != pcDstIP) && (0 == usDstPort))
    {
        memset(&stAddr, 0, sizeof(stAddr));
        stAddr.sin_family = AF_INET;
        stAddr.sin_port = htons(usDstPort);
        stAddr.sin_addr.s_addr = inet_addr(pcDstIP);

        addrlen = sizeof(stAddr);
        pstAddr = &stAddr;
    }

    iRet = __yc_socket_send(iSocketID, pData, uiDataLen, 0, (struct sockaddr *)pstAddr, addrlen);

    return iRet;
}

/* ֵ>0Ϊյݳȣ<=0ʾʧ */
int yc_socket_recv(IN int iSocketID,  IN uint uiWaitTime,
                   OUT void *pBuffer, IN uint uiBufferLen,
                   OUT char *pcSrcIP, OUT ushort *pusSrcPort)
{
    int iRet = YC_ERROR_FAILED;
    socklen_t addrlen = 0;
    struct sockaddr_in stAddr;

    addrlen = sizeof(stAddr);
    memset(&stAddr, 0, sizeof(stAddr));
    iRet = __yc_select_recv(iSocketID, uiWaitTime, pBuffer, uiBufferLen, 0,
                            (struct sockaddr *)&stAddr, &addrlen);

    if (iRet > 0)
    {
        if (NULL != pcSrcIP)
        {
            strcpy(pcSrcIP, inet_ntoa(stAddr.sin_addr));
        }

        if (NULL != pusSrcPort)
        {
            *pusSrcPort = ntohs(stAddr.sin_port);
        }
    }

    return iRet;
}

int yc_socket_set_blocking(IN int iSocketID,  IN bool_t bIsBlock)
{
	int val;
	int err;

	val = fcntl(iSocketID, F_GETFL, 0);

	if (bIsBlock)
        err = fcntl(iSocketID, F_SETFL, val & ~O_NONBLOCK);
	else
        err = fcntl(iSocketID, F_SETFL, val | O_NONBLOCK);

    if (-1 == err)
        err = YC_ERROR_FAILED;
    else
        err = YC_ERROR_SUCCESS;

	return err;
}

int yc_socket_connect(IN const char *pcDstIP,
                      IN ushort usDstPort,
                      IN uint uiWaitTime)
{
    int iRet = YC_ERROR_FAILED;
    int iSocketFd = -1;
    struct sockaddr_in stAddr;

    iSocketFd = socket(AF_INET, SOCK_STREAM, 0);
    if (-1 == iSocketFd)
    {
        return YC_ERROR_FAILED;
    }

    iRet = yc_socket_set_blocking(iSocketFd, true);
    if (YC_ERROR_SUCCESS != iRet)
    {
        close(iSocketFd);
        return YC_ERROR_FAILED;
    }

    memset(&stAddr, 0, sizeof(stAddr));
    stAddr.sin_family = AF_INET;
    stAddr.sin_port = htons(usDstPort);
    stAddr.sin_addr.s_addr = inet_addr(pcDstIP);
    iRet = connect(iSocketFd, (struct sockaddr *)&stAddr, sizeof(stAddr));
    iRet = yc_socket_check_connect(iSocketFd, iRet, uiWaitTime);

    if (YC_ERROR_SUCCESS == iRet)
        iRet = yc_socket_set_blocking(iSocketFd, false);

    if (YC_ERROR_SUCCESS != iRet)
    {
        close(iSocketFd);
        iSocketFd = -1;
    }

    return iSocketFd;
}

yc_socket_server_s *yc_udp_server_create(IN ushort usLocalPort,
                                         IN yc_udp_server_rcv_pf pfRcvCallback,
                                         IN void *pArg)
{
    int iRet = YC_ERROR_FAILED;
    int iSocketFd = -1;
    int iQuefd = -1;
    int iReuse = 1;
    struct sockaddr_in stAddr;
    char acQueName[SKT_QUEUE_NAME_LEN];
	yc_socket_server_s *server_handle = NULL;
	yc_socket_server_ctx_s *server_handle_ctx = NULL;

	server_handle = yc_malloc(sizeof(yc_socket_server_s));
    if (NULL == server_handle)
    {
		return NULL;
    }

    server_handle_ctx = yc_malloc(sizeof(yc_socket_server_ctx_s));
    if (NULL == server_handle_ctx)
    {
        yc_free(server_handle);
		return NULL;
    }

    acQueName[0] = '\0';
    sprintf(acQueName, "que-port-%hu", usLocalPort);
    iQuefd = yc_queue_create(acQueName, 3);
    if (iQuefd < 0)
    {
        yc_free(server_handle);
        yc_free(server_handle_ctx);
        return NULL;
    }

    iSocketFd = socket(AF_INET, SOCK_DGRAM, 0);
    if (-1 == iSocketFd)
    {
        yc_free(server_handle);
        yc_free(server_handle_ctx);
        yc_queue_destroy(iQuefd);
        return NULL;
    }

    iRet = setsockopt(iSocketFd, SOL_SOCKET, SO_REUSEADDR,
                      (const void *)&iReuse, sizeof(iReuse));
    if (0 != iRet)
    {
        yc_free(server_handle);
        yc_free(server_handle_ctx);
        yc_queue_destroy(iQuefd);
        close(iSocketFd);
        return NULL;
    }

    memset(&stAddr, 0, sizeof(stAddr));
    stAddr.sin_family = AF_INET;
    stAddr.sin_port = htons(usLocalPort);
    stAddr.sin_addr.s_addr = INADDR_ANY;

	iRet = bind(iSocketFd, (struct sockaddr *)&stAddr, sizeof(stAddr));
	if (0 != iRet)
	{
	    yc_free(server_handle);
        yc_free(server_handle_ctx);
        yc_queue_destroy(iQuefd);
		close(iSocketFd);
		return NULL;
	}

    memset(server_handle, 0, sizeof(yc_socket_server_s));
    memset(server_handle_ctx, 0, sizeof(yc_socket_server_ctx_s));
    server_handle->iSockfd = iSocketFd;
    server_handle_ctx->iQuefd = iQuefd;
    server_handle_ctx->pfRcvCallback = pfRcvCallback;
    server_handle_ctx->pArg = pArg;
    server_handle->ctx = server_handle_ctx;

    iRet = pthread_create(&(server_handle_ctx->stThreadID), NULL, udp_server_rcv_thread, server_handle);
    if (0 != iRet)
	{
	    yc_free(server_handle);
        yc_free(server_handle_ctx);
        yc_queue_destroy(iQuefd);
		close(iSocketFd);
		return NULL;
	}

    return server_handle;
}

int yc_udp_server_destroy(IN yc_socket_server_s *pstHandle)
{
    int iRet   = YC_ERROR_FAILED;
    yc_socket_msg_s stMsg;

    memset(&stMsg, 0, sizeof(stMsg));
    stMsg.msg = YC_SOCKET_SERVER_CLOSE_MSG;

    iRet = yc_queue_write(pstHandle->ctx->iQuefd, &stMsg, sizeof(stMsg));
    if (YC_ERROR_SUCCESS != iRet)
        return iRet;

    iRet = pthread_join(pstHandle->ctx->stThreadID, NULL);
    if (0 != iRet)
        return YC_ERROR_FAILED;

    yc_queue_destroy(pstHandle->ctx->iQuefd);
    close(pstHandle->iSockfd);
    yc_free(pstHandle->ctx);
    yc_free(pstHandle);

    return YC_ERROR_SUCCESS;
}

yc_socket_server_s *yc_tcp_server_create(IN unsigned short usLocalPort,
                                      IN tcp_server_rcv_pf pfRcvCallback,
                                      IN void *pArg)
{
    int iRet = YC_ERROR_FAILED;
    int iSocketFd = -1;
    int iQuefd = -1;
    int iReuse = 1;
    struct sockaddr_in stAddr;
    char acQueName[SKT_QUEUE_NAME_LEN];
	yc_socket_server_s *server_handle = NULL;
	yc_socket_server_ctx_s *server_handle_ctx = NULL;

	server_handle = yc_malloc(sizeof(yc_socket_server_s));
    if (NULL == server_handle)
    {
		return NULL;
    }

    server_handle_ctx = yc_malloc(sizeof(yc_socket_server_ctx_s));
    if (NULL == server_handle_ctx)
    {
        yc_free(server_handle);
		return NULL;
    }

    acQueName[0] = '\0';
    sprintf(acQueName, "que-port-%hu", usLocalPort);
    iQuefd = yc_queue_create(acQueName, 0);
    if (iQuefd < 0)
    {
        yc_free(server_handle);
        yc_free(server_handle_ctx);
        return NULL;
    }


    iSocketFd = socket(AF_INET, SOCK_STREAM, 0);
    if (-1 == iSocketFd)
    {
        yc_free(server_handle);
        yc_free(server_handle_ctx);
        yc_queue_destroy(iQuefd);
        return NULL;
    }

    iRet = setsockopt(iSocketFd, SOL_SOCKET, SO_REUSEADDR,
                      (const void *)&iReuse, sizeof(iReuse));
    if (0 != iRet)
    {
        yc_free(server_handle);
        yc_free(server_handle_ctx);
        yc_queue_destroy(iQuefd);
        close(iSocketFd);
        return NULL;
    }

    memset(&stAddr, 0, sizeof(stAddr));
    stAddr.sin_family = AF_INET;
    stAddr.sin_port = htons(usLocalPort);
    stAddr.sin_addr.s_addr = INADDR_ANY;

	iRet = bind(iSocketFd, (struct sockaddr *)&stAddr, sizeof(stAddr));
	if (0 != iRet)
	{
        yc_free(server_handle);
        yc_free(server_handle_ctx);
        yc_queue_destroy(iQuefd);
        close(iSocketFd);
        return NULL;
	}

	iRet = listen(iSocketFd, SOMAXCONN);
	if (0 != iRet)
	{
        yc_free(server_handle);
        yc_free(server_handle_ctx);
        yc_queue_destroy(iQuefd);
        close(iSocketFd);
        return NULL;
	}

	memset(server_handle, 0, sizeof(yc_socket_server_s));
    memset(server_handle_ctx, 0, sizeof(yc_socket_server_ctx_s));
    server_handle->iSockfd = iSocketFd;
    server_handle_ctx->iQuefd = iQuefd;
    server_handle_ctx->pfRcvCallback = pfRcvCallback;
    server_handle_ctx->pArg = pArg;
    INIT_LIST_HEAD(&server_handle_ctx->stNode);
    iRet = pthread_rwlock_init(&server_handle_ctx->stClientLock, NULL);
    if (0 != iRet)
    {
        yc_debug_print(YC_DEBUG_TYPE_ERROR, "server %d init lock failed.", iSocketFd);
    }
    server_handle->ctx = server_handle_ctx;

    iRet = pthread_create(&(server_handle_ctx->stThreadID), NULL, tcp_server_accept_thread, server_handle);
    if (0 != iRet)
	{
        iRet = pthread_rwlock_destroy(&server_handle_ctx->stClientLock);
        if (0 != iRet)
        {
            yc_debug_print(YC_DEBUG_TYPE_ERROR, "server %d destroy lock failed.", iSocketFd);
        }
        yc_free(server_handle);
        yc_free(server_handle_ctx);
        yc_queue_destroy(iQuefd);
        close(iSocketFd);
        return NULL;
	}

    return server_handle;
}

int yc_tcp_server_destroy(IN yc_socket_server_s *pstHandle)
{
    int iRet = YC_ERROR_FAILED;
    yc_socket_msg_s stMsg;

    memset(&stMsg, 0, sizeof(stMsg));
    stMsg.msg = YC_SOCKET_SERVER_CLOSE_MSG;

    iRet = yc_queue_write(pstHandle->ctx->iQuefd, &stMsg, sizeof(stMsg));
    if (YC_ERROR_SUCCESS != iRet)
        return iRet;

    iRet = pthread_join(pstHandle->ctx->stThreadID, NULL);
    if (0 != iRet)
        return YC_ERROR_FAILED;

    iRet = pthread_rwlock_destroy(&pstHandle->ctx->stClientLock);
    if (0 != iRet)
    {
        yc_debug_print(YC_DEBUG_TYPE_ERROR, "server %d destroy lock failed.", pstHandle->iSockfd);
    }
    yc_queue_destroy(pstHandle->ctx->iQuefd);
    close(pstHandle->iSockfd);
    yc_free(pstHandle->ctx);
    yc_free(pstHandle);

    return YC_ERROR_SUCCESS;
}

int yc_tcp_server_close_client(IN yc_socket_server_s *pstHandle,
                               IN int iClientFd)
{
    yc_socket_msg_s stMsg;
    int iRet;

    memset(&stMsg, 0, sizeof(stMsg));
    stMsg.msg = YC_SOCKET_CLOSE_CLIENT_MSG;
    stMsg.iSockfd = iClientFd;

    iRet = yc_queue_write(pstHandle->ctx->iQuefd, &stMsg, sizeof(stMsg));

    return iRet;
}

void yc_tcp_server_show_client(IN const yc_socket_server_s *pstHandle)
{
    int iRet;
    uint uiID = 0;
    socklen_t iAddrLen;
    struct sockaddr_in stAddr;
    yc_socket_client_s *client = NULL;

    printf("all client:\r\n");

    pthread_rwlock_rdlock(&pstHandle->ctx->stClientLock);
    list_for_each_entry(client, &pstHandle->ctx->stNode, stNode)
    {
        memset(&stAddr, 0, sizeof(stAddr));
		iAddrLen = sizeof(stAddr);
        iRet = getpeername(client->iSockfd, (struct sockaddr *)&stAddr, &iAddrLen);
        if (0 == iRet)
        {
            printf("%u: %s : %hu\r\n", ++uiID, inet_ntoa(stAddr.sin_addr), ntohs(stAddr.sin_port));
        }
        else
        {
            printf("%u: client %d error\r\n", ++uiID, client->iSockfd);
        }
    }
    pthread_rwlock_unlock(&pstHandle->ctx->stClientLock);

    return;
}

int yc_unix_socket_send(IN int iSocketID, IN const char *pcName,
                        IN const void *pData, IN uint uiDataLen)
{
    int iRet = YC_ERROR_FAILED;
    int iAddrLen = 0;
    struct sockaddr_un stAddr;

    if (NULL != pcName)
    {
        memset(&stAddr, 0, sizeof(stAddr));
        stAddr.sun_family = AF_UNIX;
        sprintf(stAddr.sun_path, YC_UNIX_DGRAM_REMOTE_NAME, pcName);
        iAddrLen = offsetof(struct sockaddr_un, sun_path) + strlen(stAddr.sun_path);

        iRet = __yc_socket_send(iSocketID, pData, uiDataLen, 0,
                                (struct sockaddr *)&stAddr, iAddrLen);
    }
    else
    {
        iRet = __yc_socket_send(iSocketID, pData, uiDataLen, 0, NULL, 0);
    }

    return iRet;
}

int yc_unix_socket_recv(IN int iSocketID,  IN uint uiWaitTime,
                        OUT void *pBuffer, IN uint uiBufferLen,
                        OUT char *pcSrcName)
{
    int iRet = YC_ERROR_FAILED;
    socklen_t addrlen = 0;
    struct sockaddr_un stAddr;
    char szSrcName[YC_UNIX_NAME_MAX_LEN];

    addrlen = sizeof(stAddr);
    memset(&stAddr, 0, sizeof(stAddr));
    iRet = __yc_select_recv(iSocketID, uiWaitTime, pBuffer, uiBufferLen, 0,
                            (struct sockaddr *)&stAddr, &addrlen);

    if (iRet > 0)
    {
        if (NULL != pcSrcName)
        {
            addrlen -= offsetof(struct sockaddr_un, sun_path);
		    stAddr.sun_path[addrlen] = '\0';
		    szSrcName[0] = '\0';
		    sscanf(stAddr.sun_path, "/var/tmp/%s.socket", szSrcName);
		    szSrcName[strlen(szSrcName) - strlen(".socket")] = '\0';
		    strcpy(pcSrcName, szSrcName);
        }
    }

    return iRet;
}

int yc_unix_dgram_send(IN const char *pcName,
                       IN const void *pData, IN uint uiDataLen)
{
    int iRet = YC_ERROR_FAILED;
    int iSocketFd = -1;
    int iAddrLen = 0;
    struct sockaddr_un stAddr;
    struct sockaddr_un stOurAddr;
    int iRandomNum = 0;
    char szOwnerName[RCV_BUF_MAX_LEN];

    szOwnerName[0] = '\0';
    iRet = unix_get_owner_name(szOwnerName);
    if (YC_ERROR_SUCCESS != iRet)
    {
        return YC_ERROR_FAILED;
    }

    iSocketFd = socket(AF_UNIX, SOCK_DGRAM, 0);
    if (-1 == iSocketFd)
    {
        return YC_ERROR_FAILED;
    }

    srand((unsigned int)time(NULL));
    iRandomNum = rand() % 99999;/* 0 ~ 99999Χ */

    memset(&stOurAddr, 0, sizeof(stOurAddr));
    stOurAddr.sun_family = AF_UNIX;
    sprintf(stOurAddr.sun_path, YC_UNIX_DGRAM_LOCAL_TEMP_NAME, szOwnerName, iRandomNum);
    iAddrLen = offsetof(struct sockaddr_un, sun_path) + strlen(stOurAddr.sun_path);

    unlink(stOurAddr.sun_path);
	iRet = bind(iSocketFd, (struct sockaddr *)&stOurAddr, iAddrLen);
	if (0 != iRet)
	{
		close(iSocketFd);
		return YC_ERROR_FAILED;
	}

    memset(&stAddr, 0, sizeof(stAddr));
    stAddr.sun_family = AF_UNIX;
    sprintf(stAddr.sun_path, YC_UNIX_DGRAM_REMOTE_NAME, pcName);
    iAddrLen = offsetof(struct sockaddr_un, sun_path) + strlen(stAddr.sun_path);

    iRet = __yc_socket_send(iSocketFd, pData, uiDataLen, 0,
                            (struct sockaddr *)&stAddr, iAddrLen);

    close(iSocketFd);
    unlink(stOurAddr.sun_path);

    return iRet;
}

int yc_unix_dgram_send_and_rcv(IN const char *pcName,  IN uint uiTimeOut,
                               IN const void *pInData, IN uint uiDataLen,
                               OUT void *pOutData,     INOUT uint *puiOutDataLen)
{
    int iRet = YC_ERROR_FAILED;
    int iSocketFd = -1;
    struct sockaddr_un stAddr;
    struct sockaddr_un stOurAddr;
    socklen_t iAddrLen;
    int iRandomNum = 0;
    char szOwnerName[RCV_BUF_MAX_LEN];

    if (NULL == puiOutDataLen)
        return YC_ERROR_PARAM;

    szOwnerName[0] = '\0';
    iRet = unix_get_owner_name(szOwnerName);
    if (YC_ERROR_SUCCESS != iRet)
    {
        return YC_ERROR_FAILED;
    }

    iSocketFd = socket(AF_UNIX, SOCK_DGRAM, 0);
    if (-1 == iSocketFd)
    {
        return YC_ERROR_FAILED;
    }

    srand((unsigned int)time(NULL));
    iRandomNum = rand() % 99999;/* 0~99999Χ */

    memset(&stOurAddr, 0, sizeof(stOurAddr));
    stOurAddr.sun_family = AF_UNIX;
    sprintf(stOurAddr.sun_path, YC_UNIX_DGRAM_LOCAL_TEMP_NAME, szOwnerName, iRandomNum);
    iAddrLen = offsetof(struct sockaddr_un, sun_path) + strlen(stOurAddr.sun_path);

    unlink(stOurAddr.sun_path);
	iRet = bind(iSocketFd, (struct sockaddr *)&stOurAddr, iAddrLen);
	if (0 != iRet)
	{
		close(iSocketFd);
		return YC_ERROR_FAILED;
	}

    memset(&stAddr, 0, sizeof(stAddr));
    stAddr.sun_family = AF_UNIX;
    sprintf(stAddr.sun_path, YC_UNIX_DGRAM_REMOTE_NAME, pcName);
    iAddrLen = offsetof(struct sockaddr_un, sun_path) + strlen(stAddr.sun_path);

    iRet = __yc_socket_send(iSocketFd, pInData, uiDataLen, 0,
                            (struct sockaddr *)&stAddr, iAddrLen);

    if (YC_ERROR_SUCCESS == iRet)
    {
        iRet = __yc_select_recv(iSocketFd, uiTimeOut, pOutData, *puiOutDataLen,
                                0, NULL, NULL);
        if (iRet > 0)
        {
            *puiOutDataLen = iRet;

            iRet = YC_ERROR_SUCCESS;
        }
    }

    close(iSocketFd);
    unlink(stOurAddr.sun_path);

    return iRet;
}

int yc_unix_stream_send(IN const char *pcName,
                        IN const void *pData, IN uint uiDataLen)
{
    int iRet = YC_ERROR_FAILED;
    int iSocketFd = -1;
    int iAddrLen = 0;
    struct sockaddr_un stAddr;
    struct sockaddr_un stOurAddr;
    int iRandomNum = 0;
    char szOwnerName[RCV_BUF_MAX_LEN];

    szOwnerName[0] = '\0';
    iRet = unix_get_owner_name(szOwnerName);
    if (YC_ERROR_SUCCESS != iRet)
    {
        return YC_ERROR_FAILED;
    }

    iSocketFd = socket(AF_UNIX, SOCK_STREAM, 0);
    if (-1 == iSocketFd)
    {
        return YC_ERROR_FAILED;
    }

    srand((unsigned int)time(NULL));
    iRandomNum = rand() % 99999;/* 0 ~ 99999Χ */

    memset(&stOurAddr, 0, sizeof(stOurAddr));
    stOurAddr.sun_family = AF_UNIX;
    sprintf(stOurAddr.sun_path, YC_UNIX_DGRAM_LOCAL_TEMP_NAME, szOwnerName, iRandomNum);
    iAddrLen = offsetof(struct sockaddr_un, sun_path) + strlen(stOurAddr.sun_path);

    unlink(stOurAddr.sun_path);
	iRet = bind(iSocketFd, (struct sockaddr *)&stOurAddr, iAddrLen);
	if (0 != iRet)
	{
		close(iSocketFd);
		return YC_ERROR_FAILED;
	}

    memset(&stAddr, 0, sizeof(stAddr));
    stAddr.sun_family = AF_UNIX;
    sprintf(stAddr.sun_path, YC_UNIX_DGRAM_REMOTE_NAME, pcName);
    iAddrLen = offsetof(struct sockaddr_un, sun_path) + strlen(stAddr.sun_path);

    iRet = connect(iSocketFd, (struct sockaddr *)&stAddr, iAddrLen);
	if (0 != iRet)
	{
		close(iSocketFd);
		return YC_ERROR_FAILED;
	}

    iRet = __yc_socket_send(iSocketFd, pData, uiDataLen, 0, NULL, 0);

    close(iSocketFd);
    unlink(stOurAddr.sun_path);

    return iRet;
}

int yc_unix_stream_send_and_rcv(IN const char *pcName,  IN uint uiTimeOut,
                                IN const void *pInData, IN uint uiDataLen,
                                OUT void *pOutData,     INOUT uint *puiOutDataLen)
{
    int iRet = YC_ERROR_FAILED;
    int iSocketFd = -1;
    struct sockaddr_un stAddr;
    struct sockaddr_un stOurAddr;
    socklen_t iAddrLen;
    int iRandomNum = 0;
    char szOwnerName[RCV_BUF_MAX_LEN];

    if (NULL == puiOutDataLen)
        return YC_ERROR_PARAM;

    szOwnerName[0] = '\0';
    iRet = unix_get_owner_name(szOwnerName);
    if (YC_ERROR_SUCCESS != iRet)
    {
        return YC_ERROR_FAILED;
    }

    iSocketFd = socket(AF_UNIX, SOCK_STREAM, 0);
    if (-1 == iSocketFd)
    {
        return YC_ERROR_FAILED;
    }

    srand((unsigned int)time(NULL));
    iRandomNum = rand() % 99999;/* 0~99999Χ */

    memset(&stOurAddr, 0, sizeof(stOurAddr));
    stOurAddr.sun_family = AF_UNIX;
    sprintf(stOurAddr.sun_path, YC_UNIX_DGRAM_LOCAL_TEMP_NAME, szOwnerName, iRandomNum);
    iAddrLen = offsetof(struct sockaddr_un, sun_path) + strlen(stOurAddr.sun_path);

    unlink(stOurAddr.sun_path);
	iRet = bind(iSocketFd, (struct sockaddr *)&stOurAddr, iAddrLen);
	if (0 != iRet)
	{
		close(iSocketFd);
		return YC_ERROR_FAILED;
	}

    memset(&stAddr, 0, sizeof(stAddr));
    stAddr.sun_family = AF_UNIX;
    sprintf(stAddr.sun_path, YC_UNIX_DGRAM_REMOTE_NAME, pcName);
    iAddrLen = offsetof(struct sockaddr_un, sun_path) + strlen(stAddr.sun_path);

    iRet = connect(iSocketFd, (struct sockaddr *)&stAddr, iAddrLen);
	if (0 != iRet)
	{
		close(iSocketFd);
		return YC_ERROR_FAILED;
	}

    iRet = __yc_socket_send(iSocketFd, pInData, uiDataLen, 0, NULL, 0);

    if (YC_ERROR_SUCCESS == iRet)
    {
        iRet = __yc_select_recv(iSocketFd, uiTimeOut, pOutData, *puiOutDataLen,
                                0, NULL, NULL);
        if (iRet > 0)
        {
            *puiOutDataLen = iRet;

            iRet = YC_ERROR_SUCCESS;
        }
    }

    close(iSocketFd);
    unlink(stOurAddr.sun_path);

    return iRet;
}

yc_socket_server_s *yc_unix_dgram_server_create(IN const char *pcName,
                                                IN yc_unix_server_rcv_pf pfRcvCallback,
                                                IN void *pArg)
{
    int iRet = YC_ERROR_FAILED;
    int iSocketFd = -1;
    int iQuefd = -1;
    int iAddrLen = 0;
    struct sockaddr_un stAddr;
    char acQueName[SKT_QUEUE_NAME_LEN];
	yc_socket_server_s *server_handle = NULL;
	yc_socket_server_ctx_s *server_handle_ctx = NULL;

    server_handle = yc_malloc(sizeof(yc_socket_server_s));
    if (NULL == server_handle)
    {
		return NULL;
    }

    server_handle_ctx = yc_malloc(sizeof(yc_socket_server_ctx_s));
    if (NULL == server_handle_ctx)
    {
        yc_free(server_handle);
		return NULL;
    }

    acQueName[0] = '\0';
    sprintf(acQueName, "que-ud-%s", pcName);
    iQuefd = yc_queue_create(acQueName, 3);
    if (iQuefd < 0)
    {
        yc_free(server_handle);
        yc_free(server_handle_ctx);
        return NULL;
    }

    iSocketFd = socket(AF_UNIX, SOCK_DGRAM, 0);
    if (-1 == iSocketFd)
    {
        yc_free(server_handle);
        yc_free(server_handle_ctx);
        yc_queue_destroy(iQuefd);
        return NULL;
    }

    memset(&stAddr, 0, sizeof(stAddr));
    stAddr.sun_family = AF_UNIX;
    sprintf(stAddr.sun_path, YC_UNIX_DGRAM_REMOTE_NAME, pcName);
    iAddrLen = offsetof(struct sockaddr_un, sun_path) + strlen(stAddr.sun_path);

    unlink(stAddr.sun_path);

	iRet = bind(iSocketFd, (struct sockaddr *)&stAddr, iAddrLen);
	if (0 != iRet)
	{
	    yc_free(server_handle);
        yc_free(server_handle_ctx);
        yc_queue_destroy(iQuefd);
		close(iSocketFd);
		return NULL;
	}

    memset(server_handle, 0, sizeof(yc_socket_server_s));
    memset(server_handle_ctx, 0, sizeof(yc_socket_server_ctx_s));
    server_handle->iSockfd = iSocketFd;
    server_handle_ctx->iQuefd = iQuefd;
    server_handle_ctx->pfRcvCallback = pfRcvCallback;
    server_handle_ctx->pArg = pArg;
    server_handle->ctx = server_handle_ctx;

    iRet = pthread_create(&(server_handle_ctx->stThreadID), NULL, unix_dgram_server_rcv_thread, server_handle);
    if (0 != iRet)
	{
	    yc_free(server_handle);
        yc_free(server_handle_ctx);
        yc_queue_destroy(iQuefd);
		close(iSocketFd);
		return NULL;
	}

    return server_handle;
}

int yc_unix_dgram_server_destroy(IN yc_socket_server_s *pstHandle)
{
    return yc_udp_server_destroy(pstHandle);
}

yc_socket_server_s *yc_unix_stream_server_create(IN const char *pcName,
                                                 IN yc_unix_server_rcv_pf pfRcvCallback,
                                                 IN void *pArg)
{
    int iRet = YC_ERROR_FAILED;
    int iSocketFd = -1;
    int iQuefd = -1;
    int iAddrLen = 0;
    struct sockaddr_un stAddr;
    char acQueName[SKT_QUEUE_NAME_LEN];
	yc_socket_server_s *server_handle = NULL;
	yc_socket_server_ctx_s *server_handle_ctx = NULL;

    server_handle = yc_malloc(sizeof(yc_socket_server_s));
    if (NULL == server_handle)
    {
		return NULL;
    }

    server_handle_ctx = yc_malloc(sizeof(yc_socket_server_ctx_s));
    if (NULL == server_handle_ctx)
    {
        yc_free(server_handle);
		return NULL;
    }

    acQueName[0] = '\0';
    sprintf(acQueName, "que-us-%s", pcName);
    iQuefd = yc_queue_create(acQueName, 3);
    if (iQuefd < 0)
    {
        yc_free(server_handle);
        yc_free(server_handle_ctx);
        return NULL;
    }

    iSocketFd = socket(AF_UNIX, SOCK_STREAM, 0);
    if (-1 == iSocketFd)
    {
        yc_free(server_handle);
        yc_free(server_handle_ctx);
        yc_queue_destroy(iQuefd);
        return NULL;
    }

    memset(&stAddr, 0, sizeof(stAddr));
    stAddr.sun_family = AF_UNIX;
    sprintf(stAddr.sun_path, YC_UNIX_DGRAM_REMOTE_NAME, pcName);
    iAddrLen = offsetof(struct sockaddr_un, sun_path) + strlen(stAddr.sun_path);

    unlink(stAddr.sun_path);

	iRet = bind(iSocketFd, (struct sockaddr *)&stAddr, iAddrLen);
	if (0 != iRet)
	{
	    yc_free(server_handle);
        yc_free(server_handle_ctx);
        yc_queue_destroy(iQuefd);
		close(iSocketFd);
		return NULL;
	}

	iRet = listen(iSocketFd, SOMAXCONN);
	if (0 != iRet)
	{
	    yc_free(server_handle);
        yc_free(server_handle_ctx);
        yc_queue_destroy(iQuefd);
		close(iSocketFd);
		return NULL;
	}

    memset(server_handle, 0, sizeof(yc_socket_server_s));
    memset(server_handle_ctx, 0, sizeof(yc_socket_server_ctx_s));
    server_handle->iSockfd = iSocketFd;
    server_handle_ctx->iQuefd = iQuefd;
    server_handle_ctx->pfRcvCallback = pfRcvCallback;
    server_handle_ctx->pArg = pArg;
    INIT_LIST_HEAD(&server_handle_ctx->stNode);
    iRet = pthread_rwlock_init(&server_handle_ctx->stClientLock, NULL);
    if (0 != iRet)
    {
        yc_debug_print(YC_DEBUG_TYPE_ERROR, "server %d init lock failed.", iSocketFd);
    }
    server_handle->ctx = server_handle_ctx;

    iRet = pthread_create(&(server_handle_ctx->stThreadID), NULL, unix_stream_server_accept_thread, server_handle);
    if (0 != iRet)
	{
	    iRet = pthread_rwlock_destroy(&server_handle_ctx->stClientLock);
        if (0 != iRet)
        {
            yc_debug_print(YC_DEBUG_TYPE_ERROR, "server %d destroy lock failed.", iSocketFd);
        }
	    yc_free(server_handle);
        yc_free(server_handle_ctx);
        yc_queue_destroy(iQuefd);
		close(iSocketFd);
		return NULL;
	}

    return server_handle;
}

int yc_unix_stream_server_destroy(IN yc_socket_server_s *pstHandle)
{
    return yc_tcp_server_destroy(pstHandle);
}

int yc_unix_stream_server_close_client(IN yc_socket_server_s *pstHandle,
                                       IN int iClientFd)
{
    return yc_tcp_server_close_client(pstHandle, iClientFd);
}

void yc_unix_stream_server_show_client(IN const yc_socket_server_s *pstHandle)
{
    int iRet;
    uint uiID = 0;
    socklen_t iAddrLen;
    struct sockaddr_un stAddr;
    yc_socket_client_s *client = NULL;
    char szSrcName[YC_UNIX_NAME_MAX_LEN];

    printf("all client:\r\n");

    pthread_rwlock_rdlock(&pstHandle->ctx->stClientLock);
    list_for_each_entry(client, &pstHandle->ctx->stNode, stNode)
    {
        memset(&stAddr, 0, sizeof(stAddr));
		iAddrLen = sizeof(stAddr);
        iRet = getpeername(client->iSockfd, (struct sockaddr *)&stAddr, &iAddrLen);
        if (0 == iRet)
        {
            iAddrLen -= offsetof(struct sockaddr_un, sun_path);
		    stAddr.sun_path[iAddrLen] = '\0';
		    szSrcName[0] = '\0';
		    sscanf(stAddr.sun_path, YC_UNIX_DGRAM_REMOTE_NAME, szSrcName);
		    szSrcName[strlen(szSrcName) - strlen(".socket")] = '\0';
            printf("%u: %s\r\n", ++uiID, szSrcName);
        }
        else
        {
            printf("%u: client %d error\r\n", ++uiID, client->iSockfd);
        }
    }
    pthread_rwlock_unlock(&pstHandle->ctx->stClientLock);

    return;
}

