#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <stdarg.h>
#include <pthread.h>
#include <sys/mman.h>
#include <sys/types.h>

#include "../include/yc_opt.h"
#include "../include/yc_basetype.h"
#include "../include/list.h"
#include "../include/yc_debug.h"
#include "../include/yc_reg.h"


struct RegEntry{
    struct list_head stNode;
    void *pbase;
    void *pmem;
    int fd;
    uint uiLen;
};

static pthread_rwlock_t g_stRegLock = PTHREAD_RWLOCK_INITIALIZER;
static struct list_head g_stRegHead = LIST_HEAD_INIT(g_stRegHead);

uint yc_read_reg32(IN uint uiAddr)
{
    uint value;
    void *pmem;
    uint baseaddr;
    int dev_fd;
    int pagesize = 4096;
    int length;
    int pad;

    dev_fd = open("/dev/mem", O_RDWR | O_NDELAY);

    if (dev_fd < 0)
    {
        YC_DBG_PRINT("open(/dev/mem) failed for 0x%x.\r\n", uiAddr);
        return 0;
    }

    pagesize = getpagesize();

    baseaddr = uiAddr & (~(pagesize - 1));
    pad = uiAddr - pagesize * baseaddr;
    length = 4 + pad;

    pmem = mmap(NULL, length, PROT_READ | PROT_WRITE, MAP_SHARED, dev_fd, baseaddr);

    if (MAP_FAILED == pmem)
    {
        YC_DBG_PRINT("mmap failed for 0x%x.\r\n", uiAddr);
        close(dev_fd);
        return 0;
    }

    value = *((volatile uint*)((char *)pmem + pad));

    if (-1 == munmap(pmem, length))
    {
        YC_DBG_PRINT("munmap failed for 0x%x.\r\n", uiAddr);
    }
    close(dev_fd);

    return value;
}

void yc_write_reg32(IN uint uiAddr, IN uint uiValue)
{
    void *pmem;
    uint baseaddr;
    int dev_fd;
    int pagesize;
    int length;
    int pad;

    dev_fd = open("/dev/mem", O_RDWR | O_NDELAY);

    if (dev_fd < 0)
    {
        YC_DBG_PRINT("open(/dev/mem) failed for 0x%x.\r\n", uiAddr);
        return;
    }

    pagesize = getpagesize();

    baseaddr = uiAddr & (~(pagesize - 1));
    pad = uiAddr - pagesize * baseaddr;
    length = 4 + pad;

    pmem = mmap(NULL, length, PROT_READ | PROT_WRITE, MAP_SHARED, dev_fd, baseaddr);

    if (MAP_FAILED == pmem)
    {
        YC_DBG_PRINT("mmap failed for 0x%x.\r\n", uiAddr);
        close(dev_fd);
        return;
    }

    *((volatile uint*)((char *)pmem + pad)) = uiValue;

    if (-1 == munmap(pmem, length))
    {
        YC_DBG_PRINT("munmap failed for 0x%x.\r\n", uiAddr);
    }
    close(dev_fd);

    return;
}

void *yc_mmap(IN uint uiAddr, IN uint uiSize)
{
    void *pmem;
    uint baseaddr;
    int dev_fd;
    int pagesize;
    int length;
    int pad;
    struct RegEntry *pstReg;

    pstReg = yc_malloc(sizeof(struct RegEntry));
    if (NULL == pstReg)
    {
        return NULL;
    }

    dev_fd = open("/dev/mem", O_RDWR | O_NDELAY);

    if (dev_fd < 0)
    {
        YC_DBG_PRINT("open(/dev/mem) failed for 0x%x.\r\n", uiAddr);
        yc_free(pstReg);
        return NULL;
    }

    pagesize = getpagesize();

    baseaddr = uiAddr & (~(pagesize - 1));
    pad = uiAddr - pagesize * baseaddr;
    length = uiSize + pad;

    pmem = mmap(NULL, length, PROT_READ | PROT_WRITE, MAP_SHARED, dev_fd, baseaddr);

    if (MAP_FAILED == pmem)
    {
        YC_DBG_PRINT("mmap failed for 0x%x.\r\n", uiAddr);
        yc_free(pstReg);
        close(dev_fd);
        return NULL;
    }

    memset(pstReg, 0, sizeof(struct RegEntry));
    pstReg->pbase = pmem;
    pstReg->pmem = (char *)pmem + pad;
    pstReg->fd = dev_fd;
    pstReg->uiLen = length;
    pthread_rwlock_wrlock(&g_stRegLock);
    list_add(&pstReg->stNode, &g_stRegHead);
    pthread_rwlock_unlock(&g_stRegLock);

    return pstReg->pmem;
}

void yc_munmap(IN void *pMem)
{
    bool_t bFound = false;
    struct RegEntry *pstReg;
    struct RegEntry *pstRegNext;

    pthread_rwlock_wrlock(&g_stRegLock);
    list_for_each_entry_safe(pstReg, pstRegNext, &g_stRegHead, stNode)
    {
        if (pMem == pstReg->pmem)
        {
            list_del(&pstReg->stNode);
            bFound = true;
            break;
        }
    }
    pthread_rwlock_unlock(&g_stRegLock);

    if (true == bFound)
    {
        if (-1 == munmap(pstReg->pbase, pstReg->uiLen))
        {
            YC_DBG_PRINT("munmap failed for 0x%p.\r\n", pMem);
        }
        close(pstReg->fd);
        yc_free(pstReg);
    }

    return;
}

