#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <stdarg.h>
#include <pthread.h>
#include <sys/un.h>
#include <sys/socket.h>

#include "../include/yc_opt.h"
#if !YC_LINUX_LITE
#include <sys/eventfd.h>
#endif

#include "../include/yc_basetype.h"
#include "../include/list.h"
#include "../include/yc_queue.h"
#include "../include/yc_debug.h"

/*eventfdʼֵΪ0ͻᴥһ¼ */
#define QUEUE_INIT_VALUE    0

typedef struct tagQueue_Msg{
    struct list_head stList;
    void *pData;
    unsigned int uiLen;
}QUEUE_MSG_S;

typedef struct tagQueue_List{
    struct list_head stList;
    struct list_head stMsg;
    int iFd;
    char *pcName;
    unsigned int uiUsed;
    unsigned int uiSize;
}QUEUE_LIST_S;

static pthread_rwlock_t gstRwLock = PTHREAD_RWLOCK_INITIALIZER;/* Ŀǰļе */
static struct list_head gstQueueHead = LIST_HEAD_INIT(gstQueueHead);

#if YC_LINUX_LITE
static int yc_queue_create_skt(const char *name)
{
    int iRet = YC_ERROR_FAILED;
    int iSocketFd = -1;
    int iAddrLen = 0;
    struct sockaddr_un stAddr;

    iSocketFd = socket(AF_UNIX, SOCK_DGRAM, 0);
    if (-1 == iSocketFd)
    {
        return -1;
    }

    memset(&stAddr, 0, sizeof(stAddr));
    stAddr.sun_family = AF_UNIX;
    sprintf(stAddr.sun_path, YC_UNIX_DGRAM_REMOTE_NAME, name);
    iAddrLen = offsetof(struct sockaddr_un, sun_path) + strlen(stAddr.sun_path);

    unlink(stAddr.sun_path);

	iRet = bind(iSocketFd, (struct sockaddr *)&stAddr, iAddrLen);
	if (0 != iRet)
	{
		close(iSocketFd);
		return -1;
	}

    return iSocketFd;
}
#endif

static void queue_del_all_msg(IN QUEUE_LIST_S *pstQueue)
{
    QUEUE_MSG_S *pstMsg = NULL;
    QUEUE_MSG_S *pstMsgNext = NULL;

    list_for_each_entry_safe(pstMsg, pstMsgNext, &pstQueue->stMsg, stList)
    {
        list_del(&pstMsg->stList);
        yc_free(pstMsg->pData);
        yc_free(pstMsg);
        pstQueue->uiUsed--;
    }

    return;
}

int yc_queue_create(IN const char *pcName, IN uint uiSize)
{
    int iFd = -1;
    char *pcQueName = NULL;
    QUEUE_LIST_S *pstQueue = NULL;

    pstQueue = yc_malloc(sizeof(QUEUE_LIST_S));
    if (NULL == pstQueue)
    {
        return YC_ERROR_MEM;
    }

    if (NULL != pcName)
    {
        pcQueName = yc_malloc(strlen(pcName) + 1);
        if (NULL == pcQueName)
        {
            yc_free(pstQueue);
            return YC_ERROR_MEM;
        }
        memset(pcQueName, 0, strlen(pcName) + 1);
        memcpy(pcQueName, pcName, strlen(pcName) + 1);
    }

#if YC_LINUX_LITE
    iFd = yc_queue_create_skt(pcName);
#else
    iFd = eventfd(QUEUE_INIT_VALUE, EFD_NONBLOCK);
#endif
    if (-1 == iFd)
    {
        if (NULL != pcName)
            yc_free(pcQueName);
        yc_free(pstQueue);
        return YC_ERROR_FAILED;
    }

    memset(pstQueue, 0, sizeof(QUEUE_LIST_S));
    pstQueue->iFd = iFd;
    pstQueue->uiSize = uiSize;
    pstQueue->pcName = pcQueName;
    INIT_LIST_HEAD(&pstQueue->stMsg);

    pthread_rwlock_wrlock(&gstRwLock);
    list_add_tail(&pstQueue->stList, &gstQueueHead);
    pthread_rwlock_unlock(&gstRwLock);

    return iFd;
}

int yc_queue_read(IN int iFd, OUT void *pData, OUT uint *puiLen)
{
    uint64 uiValue = 0;
    bool_t bFound  = false;
    int iRet = YC_ERROR_FAILED;
    QUEUE_MSG_S *pstMsg = NULL;
    QUEUE_LIST_S *pstQueue = NULL;

    pthread_rwlock_wrlock(&gstRwLock);
    list_for_each_entry(pstQueue, &gstQueueHead, stList)
    {
        if (iFd == pstQueue->iFd)
        {
            bFound = true;
            break;
        }
    }

    if (true == bFound)
    {
        if (0 == pstQueue->uiUsed)
        {
            iRet = YC_ERROR_EMPTY;
        }
        else
        {
            pstMsg = list_first_entry(&pstQueue->stMsg, QUEUE_MSG_S, stList);
            YC_DBGASSERT(NULL != pstMsg);

            memcpy(pData, pstMsg->pData, pstMsg->uiLen);
            *puiLen = pstMsg->uiLen;

            pstQueue->uiUsed--;
            list_del(&pstMsg->stList);
            yc_free(pstMsg->pData);
            yc_free(pstMsg);

#if YC_LINUX_LITE
            iRet = yc_unix_socket_recv(iFd, 0, &uiValue, sizeof(uiValue), NULL);
            if (iRet != sizeof(uiValue))
            {
                yc_debug_print(YC_DEBUG_TYPE_INFO, "read skt error");
            }
#else
            if (0 == pstQueue->uiUsed)
            {
                iRet = read(iFd, &uiValue, sizeof(uiValue));
                if (iRet != sizeof(uiValue))
                {
                    yc_debug_print(YC_DEBUG_TYPE_INFO, "read eventfd error");
                }
            }
#endif
            iRet = YC_ERROR_SUCCESS;
        }
    }
    else
    {
        iRet = YC_ERROR_FAILED;
    }
    pthread_rwlock_unlock(&gstRwLock);

    return iRet;
}

int yc_queue_write(IN int iFd, IN const void *pData, IN uint uiLen)
{
    uint64 uiValue = 1;
    bool_t bFound  = false;
    int iRet = YC_ERROR_FAILED;
    QUEUE_MSG_S *pstMsg = NULL;
    QUEUE_LIST_S *pstQueue = NULL;

    pstMsg = yc_malloc(sizeof(QUEUE_MSG_S));
    if (NULL == pstMsg)
    {
        return YC_ERROR_MEM;
    }

    memset(pstMsg, 0, sizeof(QUEUE_MSG_S));
    pstMsg->pData = yc_malloc(uiLen);
    if (NULL == pstMsg->pData)
    {
        yc_free(pstMsg);
        return YC_ERROR_MEM;
    }

    memcpy(pstMsg->pData, pData, uiLen);
    pstMsg->uiLen = uiLen;

    pthread_rwlock_wrlock(&gstRwLock);
    list_for_each_entry(pstQueue, &gstQueueHead, stList)
    {
        if (iFd == pstQueue->iFd)
        {
            if ((0 == pstQueue->uiSize) ||
                ((0 != pstQueue->uiSize) &&
                 (pstQueue->uiUsed < pstQueue->uiSize)))
            {
                bFound = true;
                pstQueue->uiUsed++;
                list_add_tail(&pstMsg->stList, &pstQueue->stMsg);
            }
            else
            {
                iRet = YC_ERROR_FULL;
            }
            break;
        }
    }

    if (true == bFound)
    {
#if YC_LINUX_LITE
        iRet = yc_unix_dgram_send(pstQueue->pcName, &uiValue, sizeof(uiValue));
        if (YC_ERROR_SUCCESS != iRet)
#else
        iRet = write(iFd, &uiValue, sizeof(uiValue));
        if (iRet != sizeof(uiValue))
#endif
        {
            pstQueue->uiUsed--;
            list_del(&pstMsg->stList);
            yc_free(pstMsg->pData);
            yc_free(pstMsg);
            iRet = YC_ERROR_FAILED;
        }
        else
        {
            iRet = YC_ERROR_SUCCESS;
        }
    }
    else
    {
        yc_free(pstMsg->pData);
        yc_free(pstMsg);
    }
    pthread_rwlock_unlock(&gstRwLock);

    return iRet;
}

void yc_queue_destroy(IN int iFd)
{
    QUEUE_LIST_S *pstQueue = NULL;
    QUEUE_LIST_S *pstQueueNext = NULL;

    pthread_rwlock_wrlock(&gstRwLock);
    list_for_each_entry_safe(pstQueue, pstQueueNext, &gstQueueHead, stList)
    {
        if (iFd == pstQueue->iFd)
        {
            queue_del_all_msg(pstQueue);

            list_del(&pstQueue->stList);
            yc_free(pstQueue->pcName);
            yc_free(pstQueue);
            close(iFd);
            break;
        }
    }
    pthread_rwlock_unlock(&gstRwLock);

    return;
}

#if !YC_LINUX_LITE
int yc_queue_get_info(IN int iFd, OUT YC_QUEUE_INFO_S *pstInfo)
{
    int iRet = YC_ERROR_FAILED;
    QUEUE_LIST_S *pstQueue = NULL;

    YC_DBGASSERT(NULL != pstInfo);

    pthread_rwlock_wrlock(&gstRwLock);
    list_for_each_entry(pstQueue, &gstQueueHead, stList)
    {
        if (iFd == pstQueue->iFd)
        {
            if ((strlen(pstQueue->pcName) + 1) > YC_QUEUE_NAME_LEN_MAX)
            {
                memcpy(pstInfo->szNmae, pstQueue->pcName, YC_QUEUE_NAME_LEN_MAX);
                pstInfo->szNmae[YC_QUEUE_NAME_LEN_MAX - 1] = '\0';
            }
            else
            {
                memcpy(pstInfo->szNmae, pstQueue->pcName, strlen(pstQueue->pcName) + 1);
            }
            pstInfo->iFd    = pstQueue->iFd;
            pstInfo->uiSize = pstQueue->uiSize;
            pstInfo->uiUsed = pstQueue->uiUsed;
            iRet = YC_ERROR_SUCCESS;
            break;
        }
    }
    pthread_rwlock_unlock(&gstRwLock);

    return iRet;
}

int yc_queue_read_by_name(IN const char *pcName, OUT void *pData, OUT uint *puiLen)
{
    uint64 uiValue = 0;
    bool_t bFound  = false;
    int iRet = YC_ERROR_FAILED;
    QUEUE_MSG_S *pstMsg = NULL;
    QUEUE_LIST_S *pstQueue = NULL;

    YC_DBGASSERT(NULL != pcName);

    pthread_rwlock_wrlock(&gstRwLock);
    list_for_each_entry(pstQueue, &gstQueueHead, stList)
    {
        if (0 == strcmp(pstQueue->pcName, pcName))
        {
            bFound = true;
            break;
        }
    }

    if (true == bFound)
    {
        if (0 == pstQueue->uiUsed)
        {
            iRet = YC_ERROR_EMPTY;
        }
        else
        {
            pstMsg = list_first_entry(&pstQueue->stMsg, QUEUE_MSG_S, stList);
            YC_DBGASSERT(NULL != pstMsg);

            memcpy(pData, pstMsg->pData, pstMsg->uiLen);
            *puiLen = pstMsg->uiLen;

            pstQueue->uiUsed--;
            list_del(&pstMsg->stList);
            yc_free(pstMsg->pData);
            yc_free(pstMsg);

            if (0 == pstQueue->uiUsed)
            {
                iRet = read(pstQueue->iFd, &uiValue, sizeof(uiValue));
                if (iRet != sizeof(uiValue))
                {
                    yc_debug_print(YC_DEBUG_TYPE_INFO, "read eventfd error");
                }
            }

            iRet = YC_ERROR_SUCCESS;
        }
    }
    else
    {
        iRet = YC_ERROR_FAILED;
    }
    pthread_rwlock_unlock(&gstRwLock);

    return iRet;
}

int yc_queue_write_by_name(IN const char *pcName, IN const void *pData, IN uint uiLen)
{
    uint64 uiValue = 1;
    bool_t bFound  = false;
    int iRet = YC_ERROR_FAILED;
    QUEUE_MSG_S *pstMsg = NULL;
    QUEUE_LIST_S *pstQueue = NULL;

    YC_DBGASSERT(NULL != pcName);

    pstMsg = yc_malloc(sizeof(QUEUE_MSG_S));
    if (NULL == pstMsg)
    {
        return YC_ERROR_FAILED;
    }

    memset(pstMsg, 0, sizeof(QUEUE_MSG_S));
    pstMsg->pData = yc_malloc(uiLen);
    if (NULL == pstMsg->pData)
    {
        yc_free(pstMsg);
        return YC_ERROR_FAILED;
    }

    memcpy(pstMsg->pData, pData, uiLen);
    pstMsg->uiLen = uiLen;

    pthread_rwlock_wrlock(&gstRwLock);
    list_for_each_entry(pstQueue, &gstQueueHead, stList)
    {
        if (0 == strcmp(pstQueue->pcName, pcName))
        {
            if ((0 == pstQueue->uiSize) ||
                ((0 != pstQueue->uiSize) &&
                 (pstQueue->uiUsed < pstQueue->uiSize)))
            {
                bFound = true;
                pstQueue->uiUsed++;
                list_add_tail(&pstMsg->stList, &pstQueue->stMsg);
            }
            else
            {
                iRet = YC_ERROR_FULL;
            }
            break;
        }
    }

    if (true == bFound)
    {
        iRet = write(pstQueue->iFd, &uiValue, sizeof(uiValue));
        if (iRet != sizeof(uiValue))
        {
            pstQueue->uiUsed--;
            list_del(&pstMsg->stList);
            yc_free(pstMsg->pData);
            yc_free(pstMsg);
            iRet = YC_ERROR_FAILED;
        }
        else
        {
            iRet = YC_ERROR_SUCCESS;
        }
    }
    else
    {
        yc_free(pstMsg->pData);
        yc_free(pstMsg);
    }
    pthread_rwlock_unlock(&gstRwLock);

    return iRet;
}

void yc_queue_destroy_by_name(IN const char *pcName)
{
    QUEUE_LIST_S *pstQueue = NULL;
    QUEUE_LIST_S *pstQueueNext = NULL;

    YC_DBGASSERT(NULL != pcName);

    pthread_rwlock_wrlock(&gstRwLock);
    list_for_each_entry_safe(pstQueue, pstQueueNext, &gstQueueHead, stList)
    {
        if (0 == strcmp(pstQueue->pcName, pcName))
        {
            queue_del_all_msg(pstQueue);

            list_del(&pstQueue->stList);
            close(pstQueue->iFd);
            yc_free(pstQueue->pcName);
            yc_free(pstQueue);
            break;
        }
    }
    pthread_rwlock_unlock(&gstRwLock);

    return;
}

int yc_queue_get_info_by_name(IN const char *pcName, OUT YC_QUEUE_INFO_S *pstInfo)
{
    int iRet = YC_ERROR_FAILED;
    QUEUE_LIST_S *pstQueue = NULL;

    YC_DBGASSERT(NULL != pcName);

    pthread_rwlock_wrlock(&gstRwLock);
    list_for_each_entry(pstQueue, &gstQueueHead, stList)
    {
        if (0 == strcmp(pstQueue->pcName, pcName))
        {
            if ((strlen(pstQueue->pcName) + 1) > YC_QUEUE_NAME_LEN_MAX)
            {
                memcpy(pstInfo->szNmae, pstQueue->pcName, YC_QUEUE_NAME_LEN_MAX);
                pstInfo->szNmae[YC_QUEUE_NAME_LEN_MAX - 1] = '\0';
            }
            else
            {
                memcpy(pstInfo->szNmae, pstQueue->pcName, strlen(pstQueue->pcName) + 1);
            }
            pstInfo->iFd    = pstQueue->iFd;
            pstInfo->uiSize = pstQueue->uiSize;
            pstInfo->uiUsed = pstQueue->uiUsed;
            iRet = YC_ERROR_SUCCESS;
            break;
        }
    }
    pthread_rwlock_unlock(&gstRwLock);

    return iRet;
}
#endif

