#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <netdb.h>
#include <stdarg.h>
#include <sys/select.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <arpa/inet.h>
#include <netinet/in.h>

#include "../include/yc_opt.h"
#include "../include/yc_basetype.h"
#include "../include/yc_debug.h"
#include "../include/yc_httpc.h"


#define YC_HTTPC_DEFAULT_PORT           80
#define YC_HTTPC_ADDRESS_LEN            256
#define YC_HTTPC_HTML_BODY_LEN          1024
#define YC_HTTPC_HTTP_STATUS_OK         200
#define YC_HTTPC_HTTP_STATUS_OFFSET_OK  206
#define YC_HTTPC_WAIT_TIME_MAX          30
#define YC_HTTPC_USER_AGENT             "yc http client v1.0.0.820"


struct tagHttpc_private{
    int iSkt;
    uint uiRemain;
};

static int yc_httpc_parse_url(const char *szUrl, char *szAddress,
                              ushort* pusPortNum, char *pcMedia)
{
	char *prefix = "http://";
	uint prefixLength = 7;
	char* from = NULL ;
	char* to = NULL;
	uint i;
	char nextChar;

	if (strncmp(szUrl, prefix, prefixLength) != 0)
	{
		yc_debug_print(YC_DEBUG_TYPE_ERROR, "httpc URL is not of the form");
		return YC_ERROR_FAILED;
	}

	from = (char *)&szUrl[prefixLength];
	to = &szAddress[0];
	for (i = 0; i < YC_HTTPC_ADDRESS_LEN; ++i)
	{
		if (*from == '\0' || *from == ':' || *from == '/')
		{
			*to = '\0';
			break;
		}
		*to++ = *from++;
	}
	if (i == YC_HTTPC_ADDRESS_LEN)
	{
		yc_debug_print(YC_DEBUG_TYPE_ERROR, "httpc is too long");
		return YC_ERROR_FAILED;
	}

 	*pusPortNum = YC_HTTPC_DEFAULT_PORT;
	nextChar = *from;
	if (nextChar == ':')
	{
		int portNumInt;
		if (sscanf(++from, "%d", &portNumInt) != 1)
		{
			yc_debug_print(YC_DEBUG_TYPE_ERROR, "httpc No port number follows: %d",portNumInt);
			return -1;
		}
		if (portNumInt < 1 || portNumInt > 65535)
		{
			yc_debug_print(YC_DEBUG_TYPE_ERROR, "httpc Bad port number");
			return YC_ERROR_FAILED;
		}
		*pusPortNum = portNumInt;
	}

    while (*from != '\0')
    {
        if (*from == '/')
        {
            strcpy(pcMedia, from);
            break;
        }
        from++;
    }

	YC_DBG_PRINT("address is %s, portNum is %hu, media is %s\n", szAddress, *pusPortNum, pcMedia);
	return YC_ERROR_SUCCESS;
}

static int yc_httpc_make_blocking(int sock)
{
	int val;

	val = fcntl(sock, F_GETFL, 0);
	if (fcntl(sock, F_SETFL, val & ~O_NONBLOCK) == -1)
	{
		return YC_ERROR_FAILED;
	}

	return YC_ERROR_SUCCESS;
}

static int yc_httpc_make_no_blocking(int sock)
{
	int val;

	val = fcntl(sock, F_GETFL, 0);
	if (fcntl(sock, F_SETFL, val | O_NONBLOCK) == -1)
	{
		return YC_ERROR_FAILED;
	}

	return YC_ERROR_SUCCESS;
}

static int yc_httpc_check_connection(int sock, int connect_res)
{
	struct timeval tmvTimeout = {YC_HTTPC_WAIT_TIME_MAX, 0};
	int result;
	fd_set fdSet;

	FD_ZERO(&fdSet);
	FD_SET(sock, &fdSet);

	if (connect_res < 0)
	{
		if (EINPROGRESS == errno)
    	{
			result = select(sock + 1, NULL, &fdSet, NULL, &tmvTimeout);
			if (result < 0)
			{
				return YC_ERROR_FAILED;
			}
			else if (result == 0)
			{
				/* timeout */
				return YC_ERROR_FAILED;
			}
			else
			{
				int valopt = 0;
				socklen_t len = sizeof(valopt);
				if (getsockopt(sock, SOL_SOCKET, SO_ERROR, (void *) &valopt, &len) < 0)
				{
					/* failed to read delayed error */
					return YC_ERROR_FAILED;
				}
				else if (valopt)
				{
					/* delayed error = valopt */
					return YC_ERROR_FAILED;
				}
			}
        }
	}

	return YC_ERROR_SUCCESS;
}

static int yc_httpc_connect(char *address, unsigned short port)
{
	struct hostent *hp;
	struct sockaddr_in server;
	int skt = -1;
	int ret = -1;

    hp = gethostbyname(address);
    if (hp == NULL )
	{
		yc_debug_print(YC_DEBUG_TYPE_ERROR, "Cannot resolve address [%s]\n", address);
		return YC_ERROR_FAILED;
	}

	skt = socket(AF_INET, SOCK_STREAM, 0);
	if (skt < 0)
	{
		yc_debug_print(YC_DEBUG_TYPE_ERROR, "Error Opening socket\n");
		return YC_ERROR_FAILED;
	}

	ret = yc_httpc_make_no_blocking(skt);
	if (YC_ERROR_SUCCESS != ret)
	{
		yc_debug_print(YC_DEBUG_TYPE_ERROR, "make socket unblock failed.\n");
		close(skt);
		return YC_ERROR_FAILED;
	}

	memset(&server, 0, sizeof(struct sockaddr_in));
	memcpy(&(server.sin_addr), hp->h_addr, hp->h_length);
	server.sin_family = AF_INET;
	server.sin_port = htons(port);
	ret = connect(skt, (struct sockaddr*)&server, sizeof(struct sockaddr_in));
	ret = yc_httpc_check_connection(skt, ret);
    if (YC_ERROR_SUCCESS == ret)
        ret = yc_httpc_make_blocking(skt);

	if (YC_ERROR_SUCCESS != ret)
	{
		yc_debug_print(YC_DEBUG_TYPE_ERROR, "connect failed\n");
		close(skt);
		skt = YC_ERROR_FAILED;
	}

	return skt;
}

static int yc_httpc_recv(int skt, void *html, int len)
{
    int ret;
    fd_set read_set;
    struct timeval tv;

    FD_ZERO(&read_set);
    FD_SET(skt, &read_set);
    tv.tv_sec  = YC_HTTPC_WAIT_TIME_MAX;
    tv.tv_usec = 0;

    ret = select(skt + 1, &read_set, NULL, NULL, &tv);
    if (ret > 0)
    {
        if (FD_ISSET(skt, &read_set))
        {
            ret = recv(skt, html, len, 0);
            if (0 == ret)
            {
                yc_debug_print(YC_DEBUG_TYPE_ERROR, "recv closed.\n");
            }
            else if (ret < 0)
            {
                yc_debug_print(YC_DEBUG_TYPE_ERROR, "recv failed.\n");
            }
            FD_CLR(skt, &read_set);
        }
        else
        {
            ret = -1;
        }
    }

    return ret;
}

static int yc_httpc_read_by_size(int skt_num, char* buffer,
                                 uint bufferMaxSize, uint NextTCPReadSize)
{
	unsigned int totBytesToRead;
	int          curBytesToRead;
	unsigned int curBytesRead;

    if (bufferMaxSize < NextTCPReadSize)
    {
        yc_debug_print(YC_DEBUG_TYPE_ERROR, "bufferMaxSize < NextTCPReadSize\n");
        return -1;
	}

    curBytesRead   = 0;
    totBytesToRead = NextTCPReadSize;

    do
    {
        curBytesToRead = yc_httpc_recv(skt_num, buffer + curBytesRead, totBytesToRead);
        if (0 >= curBytesToRead)
            return curBytesToRead;

        curBytesRead   += curBytesToRead;
        totBytesToRead -= curBytesToRead;
    } while(totBytesToRead > 0);

	return curBytesRead;
}

static void yc_httpc_parse_line(yc_httpc_handle_s *pstHttpc,
                                char* line, unsigned short len)
{
    if (sscanf(line, "Content-Length: %u", &pstHttpc->content_len) == 1 ||
	    sscanf(line, "Content-length: %u", &pstHttpc->content_len) == 1)
	{
        yc_debug_print(YC_DEBUG_TYPE_INFO, "content len '%u'\n", pstHttpc->content_len);
	}
	else if (0 == strncmp("Accept-Ranges: bytes", line, strlen("Accept-Ranges: bytes")))
	{
        pstHttpc->is_accept_range = true;
	}
	else if (0 == strncmp("Connection: Close", line, strlen("Connection: Close")))
	{
        pstHttpc->is_long_conn = false;
	}

    return;
}

static int yc_httpc_get_response(yc_httpc_handle_s *pstHttpc,
                                 char* presponseBuffer,
                                 unsigned int responseBufferSize)
{
    int line_count, total_len = 0;
    int len;
    char *buf, *q;
    unsigned char ch;
    int ret;

    buf = yc_malloc(YC_HTTPC_HTML_BODY_LEN);
    if (!buf)
    {
        yc_debug_print(YC_DEBUG_TYPE_ERROR, "malloc recv buf failed.\n");
        return YC_ERROR_MEM;
    }

    line_count = 0;
    memset(presponseBuffer, 0, responseBufferSize);
    for(;;) {
        memset(buf, 0, YC_HTTPC_HTML_BODY_LEN);
        q = buf;
        for(;;) {
            ret = yc_httpc_recv(pstHttpc->pri->iSkt, &ch, 1);
            if (ret <= 0)
        	    break;
            if (ch == '\n')
                break;
            else if (ch != '\r') {
                if ((q - buf) < YC_HTTPC_HTML_BODY_LEN - 1)
                    *q++ = ch;
            }
        }
        *q = '\0';

        YC_DBG_PRINT("line = '%s'\r\n", buf);

        /* test if last line */
        if (buf[0] == '\0')
        {
            *(presponseBuffer + total_len) = '\0';
            break;
        }

        if (line_count == 0) {
            *q++ = '\r';
            *q   = '\n';
            total_len  = q - buf + 1;
            memcpy(presponseBuffer, buf, total_len);
        }
        else
        {
            *q++ = '\r';
            *q   = '\n';
            len = q - buf + 1;
            memcpy(presponseBuffer + total_len, buf, len);
            total_len += len;
            yc_httpc_parse_line(pstHttpc, buf, len);
        }
        line_count++;
    }

    yc_free(buf);

	return total_len;
}

static int yc_httpc_get_send_request(yc_httpc_handle_s *pstHttpc,
                                     const char *url, uint uiOffset,
                                     char *address, char *media)
{
    int ret = -1;
    int len = 0;
    char *pos;
    char *html = NULL;

    html = yc_malloc(YC_HTTPC_HTML_BODY_LEN);
    if (!html)
    {
	    return YC_ERROR_MEM;
    }

    if ((0 != strncmp(url, "http://", 7)) && (0 != strcmp(media, url)))
    {
        /* url׷ӵmediaĩβ滻ԭĵַ */
        pos = media + strlen(media);
        while (*pos != '/')
        {
            pos--;
        }
        strcpy(pos + 1, url);
    }
    else
    {
        strcpy(media, url);
    }

    memset(html, 0, YC_HTTPC_HTML_BODY_LEN);
    len  = sprintf(html, "GET %s HTTP/1.1\r\n", media);
    len += sprintf(html + len, "Host: %s\r\n", address);
    if (uiOffset)
        len += sprintf(html + len, "Range: bytes=%u-\r\n", uiOffset);
    len += sprintf(html + len, "User-Agent: %s\r\n", YC_HTTPC_USER_AGENT);
    len += sprintf(html + len, "\r\n");

    YC_DBG_PRINT("send: '%s'\r\n", html);

	ret = send(pstHttpc->pri->iSkt, html, len, 0);
    yc_free(html);

    if (ret <= 0)
    {
        yc_debug_print(YC_DEBUG_TYPE_ERROR, "send request failed.\r\n");
        return YC_ERROR_FAILED;
    }

    return YC_ERROR_SUCCESS;
}

static int yc_httpc_put_send_request(yc_httpc_handle_s *pstHttpc,
                                     const char *url,
                                     const char *pcName,
                                     uint uiDataLen,
                                     char *address, char *media)
{
    int ret = -1;
    int len = 0;
    char *pos;
    char *html = NULL;

    html = yc_malloc(YC_HTTPC_HTML_BODY_LEN);
    if (!html)
    {
	    return YC_ERROR_MEM;
    }

    if ((0 != strncmp(url, "http://", 7)) && (0 != strcmp(media, url)))
    {
        /* url׷ӵmediaĩβ滻ԭĵַ */
        pos = media + strlen(media);
        while (*pos != '/')
        {
            pos--;
        }
        strcpy(pos + 1, url);
    }
    else
    {
        strcpy(media, url);
    }

    memset(html, 0, YC_HTTPC_HTML_BODY_LEN);
    len  = sprintf(html, "PUT %s/%s HTTP/1.1\r\n", media, pcName);
    len += sprintf(html + len, "Host: %s\r\n", address);
    len += sprintf(html + len, "Content-Type: %s\r\n", "application/octet-stream");
    len += sprintf(html + len, "Content-Length: %u\r\n", uiDataLen);
    len += sprintf(html + len, "Referer: %s\r\n", "http://www.wdyichen.cn");
    //len += sprintf(html + len, "Connection: %s\r\n", "Keep-Alive");
    //len += sprintf(html + len, "Accept: */*\r\n");
    //len += sprintf(html + len, "Accept-Language: zh-cn\r\n");
    len += sprintf(html + len, "User-Agent: %s\r\n", YC_HTTPC_USER_AGENT);
    len += sprintf(html + len, "\r\n");

    YC_DBG_PRINT("send: '%s'\r\n", html);

	ret = send(pstHttpc->pri->iSkt, html, len, 0);
    yc_free(html);

    if (ret <= 0)
    {
        yc_debug_print(YC_DEBUG_TYPE_ERROR, "send request failed.\r\n");
        return YC_ERROR_FAILED;
    }

    return YC_ERROR_SUCCESS;
}

/* offsetָλأڶϵУ
   handleɽӿڴһεøָָNULL
   ɹERROR_SUCCESSֵΪʧܡ
   ע⣺ֳ֧ӵķԷãҪȹر´ */
int yc_httpc_get_request(IN const char *pcUrl,
                         IN uint uiOffset,
                         INOUT yc_httpc_handle_s **pstHttpHandle)
{
    int skt = -1;
    int iRet = YC_ERROR_FAILED;
    char *html = NULL;
    unsigned short port;
    char *address = NULL;
    char *media   =  NULL;
    unsigned int responseCode = 0;
    yc_httpc_prvi_s   *pstPriv  = NULL;
    yc_httpc_handle_s *pstHttpc = NULL;

    if (NULL == pstHttpHandle)
        return YC_ERROR_PARAM;

    if (NULL == *pstHttpHandle)
    {
        pstHttpc = yc_malloc(sizeof(yc_httpc_handle_s));
        if (NULL == pstHttpc)
            return YC_ERROR_MEM;

        pstPriv = yc_malloc(sizeof(yc_httpc_prvi_s));
        if (NULL == pstPriv)
        {
            yc_free(pstHttpc);
            return YC_ERROR_MEM;
        }

        memset(pstPriv,  0, sizeof(yc_httpc_prvi_s));
        memset(pstHttpc, 0, sizeof(yc_httpc_handle_s));

        pstHttpc->pri = pstPriv;
        pstHttpc->pri->iSkt = -1;
        pstHttpc->is_accept_range = false;
        pstHttpc->is_long_conn    = true;/* HTTP1.1ʼĬϳ */
    }
    else
    {
        pstHttpc = *pstHttpHandle;
    }

    address = yc_malloc(YC_HTTPC_ADDRESS_LEN);
	if (!address)
	{
	    if (NULL == *pstHttpHandle)
	    {
    	    yc_free(pstPriv);
            yc_free(pstHttpc);
        }
	    return YC_ERROR_MEM;
    }

	media = yc_malloc(YC_HTTPC_ADDRESS_LEN);
	if (!media)
	{
	    yc_free(address);
	    if (NULL == *pstHttpHandle)
	    {
    	    yc_free(pstPriv);
            yc_free(pstHttpc);
        }
	    return YC_ERROR_MEM;
    }

	memset(address, 0, YC_HTTPC_ADDRESS_LEN);
	memset(media,   0, YC_HTTPC_ADDRESS_LEN);

	iRet = yc_httpc_parse_url(pcUrl, address, &port, media);
	if (YC_ERROR_SUCCESS != iRet)
	{
	    yc_free(address);
	    yc_free(media);
	    if (NULL == *pstHttpHandle)
	    {
    	    yc_free(pstPriv);
            yc_free(pstHttpc);
        }
	    return YC_ERROR_FAILED;
    }

    if (NULL == *pstHttpHandle)
    {
        skt = yc_httpc_connect(address, port);
        if (skt < 0)
        {
            yc_free(address);
    	    yc_free(media);
    	    yc_free(pstPriv);
            yc_free(pstHttpc);
            return YC_ERROR_FAILED;
        }

        pstHttpc->pri->iSkt = skt;
    }

    iRet = yc_httpc_get_send_request(pstHttpc, pcUrl, uiOffset, address, media);
    if (YC_ERROR_SUCCESS != iRet)
    {
        yc_free(address);
	    yc_free(media);
	    if (NULL == *pstHttpHandle)
	    {
	        close(pstPriv->iSkt);
	        yc_free(pstPriv);
            yc_free(pstHttpc);
        }
        return iRet;
    }

    yc_free(address);
	yc_free(media);

    html = yc_malloc(YC_HTTPC_HTML_BODY_LEN);
    if (!html)
    {
        iRet = YC_ERROR_MEM;
        goto fail;
    }

    iRet = yc_httpc_get_response(pstHttpc, html, YC_HTTPC_HTML_BODY_LEN);
    if (iRet > 0)
    {
        YC_DBG_PRINT("recvd: '%s'\r\n", html);
        iRet = YC_ERROR_FAILED;

        if (sscanf(html, "%*s%u", &responseCode) != 1)
    	{
    		yc_debug_print(YC_DEBUG_TYPE_ERROR, "o response code in line\n");
    		goto fail2;
    	}

    	if ((YC_HTTPC_HTTP_STATUS_OK != responseCode) && (0 == uiOffset))
    	{
    		yc_debug_print(YC_DEBUG_TYPE_ERROR, "esponse code %u not success\n", responseCode);
            goto fail2;
    	}
    	else if ((YC_HTTPC_HTTP_STATUS_OFFSET_OK != responseCode) && (0 != uiOffset))
    	{
    		yc_debug_print(YC_DEBUG_TYPE_ERROR, "esponse code %u not success\n", responseCode);
    		iRet = YC_ERROR_NOT_SUPPORT;
            goto fail2;
    	}
    }
    else
    {
        iRet = YC_ERROR_FAILED;
        goto fail2;
    }

    yc_free(html);

    pstHttpc->pri->uiRemain = pstHttpc->content_len;
    if (NULL == *pstHttpHandle)
        *pstHttpHandle = pstHttpc;

    return YC_ERROR_SUCCESS;

fail2:
    yc_free(html);
fail:
    if (NULL == *pstHttpHandle)
    {
	    close(pstPriv->iSkt);
        yc_free(pstPriv);
        yc_free(pstHttpc);
    }
    return iRet;
}

/* ضȡݳȣ<=0Ϊʧ */
int yc_httpc_get_data(yc_httpc_handle_s *pstHttpc,
                      OUT char *pcBuf, IN uint uiBufLen)
{
    int iRet = YC_ERROR_FAILED;

    if (uiBufLen >= pstHttpc->pri->uiRemain)
    {
        iRet = yc_httpc_read_by_size(pstHttpc->pri->iSkt, pcBuf,
                                     uiBufLen, pstHttpc->pri->uiRemain);
        if (iRet > 0)
            pstHttpc->pri->uiRemain = 0;
    }
    else
    {
        iRet = yc_httpc_read_by_size(pstHttpc->pri->iSkt, pcBuf,
                                     uiBufLen, uiBufLen);

        if (iRet > 0)
            pstHttpc->pri->uiRemain -= uiBufLen;
    }

    return iRet;
}

int yc_httpc_put_request(IN const char *pcUrl,
                         IN const char *pcName,
                         IN uint uiDataLen,
                         INOUT yc_httpc_handle_s **pstHttpHandle)
{
    int skt = -1;
    int iRet = YC_ERROR_FAILED;
    unsigned short port;
    char *address = NULL;
    char *media   =  NULL;
    yc_httpc_prvi_s   *pstPriv  = NULL;
    yc_httpc_handle_s *pstHttpc = NULL;

    if (NULL == pstHttpHandle)
        return YC_ERROR_PARAM;

    if (NULL == *pstHttpHandle)
    {
        pstHttpc = yc_malloc(sizeof(yc_httpc_handle_s));
        if (NULL == pstHttpc)
            return YC_ERROR_MEM;

        pstPriv = yc_malloc(sizeof(yc_httpc_prvi_s));
        if (NULL == pstPriv)
        {
            yc_free(pstHttpc);
            return YC_ERROR_MEM;
        }

        memset(pstPriv,  0, sizeof(yc_httpc_prvi_s));
        memset(pstHttpc, 0, sizeof(yc_httpc_handle_s));

        pstHttpc->pri = pstPriv;
        pstHttpc->pri->iSkt = -1;
        pstHttpc->is_accept_range = false;
        pstHttpc->is_long_conn    = true;/* HTTP1.1ʼĬϳ */
    }
    else
    {
        pstHttpc = *pstHttpHandle;
    }

    address = yc_malloc(YC_HTTPC_ADDRESS_LEN);
	if (!address)
	{
	    if (NULL == *pstHttpHandle)
	    {
    	    yc_free(pstPriv);
            yc_free(pstHttpc);
        }
	    return YC_ERROR_MEM;
    }

	media = yc_malloc(YC_HTTPC_ADDRESS_LEN);
	if (!media)
	{
	    yc_free(address);
	    if (NULL == *pstHttpHandle)
	    {
    	    yc_free(pstPriv);
            yc_free(pstHttpc);
        }
	    return YC_ERROR_MEM;
    }

	memset(address, 0, YC_HTTPC_ADDRESS_LEN);
	memset(media,   0, YC_HTTPC_ADDRESS_LEN);

	iRet = yc_httpc_parse_url(pcUrl, address, &port, media);
	if (YC_ERROR_SUCCESS != iRet)
	{
	    yc_free(address);
	    yc_free(media);
	    if (NULL == *pstHttpHandle)
	    {
    	    yc_free(pstPriv);
            yc_free(pstHttpc);
        }
	    return YC_ERROR_FAILED;
    }

    if (NULL == *pstHttpHandle)
    {
        skt = yc_httpc_connect(address, port);
        if (skt < 0)
        {
            yc_free(address);
    	    yc_free(media);
    	    yc_free(pstPriv);
            yc_free(pstHttpc);
            return YC_ERROR_FAILED;
        }

        pstHttpc->pri->iSkt = skt;
    }

    iRet = yc_httpc_put_send_request(pstHttpc, pcUrl, pcName, uiDataLen, address, media);
    if (YC_ERROR_SUCCESS != iRet)
    {
        yc_free(address);
	    yc_free(media);
	    if (NULL == *pstHttpHandle)
	    {
	        close(pstPriv->iSkt);
	        yc_free(pstPriv);
            yc_free(pstHttpc);
        }
        return iRet;
    }

    yc_free(address);
	yc_free(media);

    pstHttpc->pri->uiRemain = uiDataLen;
    pstHttpc->content_len   = uiDataLen;

    if (NULL == *pstHttpHandle)
        *pstHttpHandle = pstHttpc;

    return YC_ERROR_SUCCESS;
}

int yc_httpc_put_data(IN yc_httpc_handle_s *pstHttpc,
                      IN const void *pData, IN uint uiDataLen)
{
    int iRet = YC_ERROR_FAILED;
    char *html = NULL;
    unsigned int responseCode = 0;

    if (pstHttpc->pri->uiRemain < uiDataLen)
    {
        return YC_ERROR_PARAM;
    }

    iRet = send(pstHttpc->pri->iSkt, pData, uiDataLen, 0);
    if (iRet > 0)
    {
        pstHttpc->pri->uiRemain -= uiDataLen;
        iRet = YC_ERROR_SUCCESS;

        if (0 == pstHttpc->pri->uiRemain)
        {
            html = yc_malloc(YC_HTTPC_HTML_BODY_LEN);
            if (!html)
            {
                return YC_ERROR_MEM;
            }

            iRet = yc_httpc_get_response(pstHttpc, html, YC_HTTPC_HTML_BODY_LEN);
            if (iRet > 0)
            {
                YC_DBG_PRINT("recvd: '%s'\r\n", html);

                if (sscanf(html, "%*s%u", &responseCode) != 1)
            	{
            		yc_debug_print(YC_DEBUG_TYPE_ERROR, "o response code in line\n");
                    yc_free(html);
            		return YC_ERROR_FAILED;
            	}

            	if (YC_HTTPC_HTTP_STATUS_OK != responseCode)
            	{
            		yc_debug_print(YC_DEBUG_TYPE_ERROR, "esponse code %u not success\n", responseCode);
                    yc_free(html);
            		return YC_ERROR_FAILED;
            	}
            }
            else
            {
                yc_free(html);
                return YC_ERROR_FAILED;
            }

            yc_free(html);
            iRet = YC_ERROR_SUCCESS;
            pstHttpc->content_len = 0;
        }
    }
    else
    {
        iRet = YC_ERROR_FAILED;
    }

    return iRet;
}

/* ر */
void yc_httpc_close(yc_httpc_handle_s *pstHttpc)
{
    close(pstHttpc->pri->iSkt);

    yc_free(pstHttpc->pri);

    yc_free(pstHttpc);

    return;
}

