#include <string.h>
#include "wm_include.h"
#include "wm_flash_map.h"
#include "wm_fwup.h"
#include "lwip/netdb.h"

#define DEMO_SSID "your ssid"
#define DEMO_PWD  "your password"
#define DEMO_URL  "http://url/image_gz.img" /* or "https://url/image_gz.img" */

#define DEMO_TASK_PRIO                  32
#define DEMO_TASK_STK_SIZE              4096

#define DEMO_OTA_ADDR_LEN               256
#define DEMO_OTA_BUF_SIZE               1024

#define DEMO_TCP_NO_BLOCK_TIME          5 /* λ:  */

#define DEMO_OTA_USE_HTTPS              0 /* ʹhttpsҪmbedtls֧ */


#define DEMO_TCP_CONNECT_NO_BLOCK       1
#define DEMO_TCP_UNTIL_WRITEABLE        1

static OS_STK demo_task_stk[DEMO_TASK_STK_SIZE / sizeof(OS_STK)];

static u32 demo_ota_session_id  = 0;
static u32 demo_ota_content_len = 0;

#if DEMO_OTA_USE_HTTPS
#include "mbedtls/debug.h"
#include "mbedtls/ssl.h"
#include "mbedtls/entropy.h"
#include "mbedtls/ctr_drbg.h"
#include "mbedtls/error.h"
#include "mbedtls/certs.h"
#include "mbedtls/net.h"

#ifdef MBEDTLS_DEBUG_C
#define DEBUG_LEVEL 0
#endif

static u8  demo_ota_is_https    = 0;

static mbedtls_entropy_context demo_ota_ssl_entropy;
static mbedtls_ctr_drbg_context demo_ota_ssl_ctr_drbg;
static mbedtls_ssl_context demo_ota_ssl;
static mbedtls_ssl_config demo_ota_ssl_conf;
#endif

static int demo_ota_download_progress(u32 download_size, u32 total_size)
{
    u32 progress = download_size * 100 / total_size;

    printf("ota upgrade progress: %u%%\r\n", progress);

	return 0;
}

#if DEMO_OTA_USE_HTTPS
#ifdef MBEDTLS_DEBUG_C
static void demo_ota_ssl_debug(void *ctx, int level,
                               const char *file, int line,
                               const char *str)
{
    printf("%s", str);
    fflush(stdout);
}
#endif

static int demo_ota_ssl_net_would_block( const mbedtls_net_context *ctx )
{
    /*
     * Never return 'WOULD BLOCK' on a non-blocking socket
     */
    if( ( fcntl( (int)(unsigned long)ctx, F_GETFL, 0 ) & O_NONBLOCK ) != O_NONBLOCK )
        return( 0 );

    switch( errno )
    {
#if defined EAGAIN
        case EAGAIN:
#endif
#if defined EWOULDBLOCK && EWOULDBLOCK != EAGAIN
        case EWOULDBLOCK:
#endif
            return( 1 );
    }
    return( 0 );
}

static int demo_ota_ssl_net_send( void *ctx, const unsigned char *buf, size_t len )
{
    int ret;
    int fd = (int)(unsigned long)ctx;

    if( fd < 0 )
        return( MBEDTLS_ERR_NET_INVALID_CONTEXT );

    ret = send( fd, buf, len, 0 );

    if( ret < 0 )
    {
        if( demo_ota_ssl_net_would_block( ctx ) != 0 )
            return( MBEDTLS_ERR_SSL_WANT_WRITE );

        if( errno == EPIPE || errno == ECONNRESET )
            return( MBEDTLS_ERR_NET_CONN_RESET );

        if( errno == EINTR )
            return( MBEDTLS_ERR_SSL_WANT_WRITE );

        return( MBEDTLS_ERR_NET_SEND_FAILED );
    }

    return( ret );
}

static int demo_ota_ssl_net_recv( void *ctx, unsigned char *buf, size_t len )
{
    int ret;
    int fd = (int)(unsigned long)ctx;

    if( fd < 0 )
        return( MBEDTLS_ERR_NET_INVALID_CONTEXT );

    ret = (int) recv( fd, buf, len, 0 );

    if( ret < 0 )
    {
        if( demo_ota_ssl_net_would_block( ctx ) != 0 )
            return( MBEDTLS_ERR_SSL_WANT_READ );

        if( errno == EPIPE || errno == ECONNRESET )
            return( MBEDTLS_ERR_NET_CONN_RESET );

        if( errno == EINTR )
            return( MBEDTLS_ERR_SSL_WANT_READ );

        return( MBEDTLS_ERR_NET_RECV_FAILED );
    }

    return( ret );
}

static int demo_ota_ssl_connect(int skt)
{
    int ret;
    const char *pers = "w600_ota";

    mbedtls_ssl_init(&demo_ota_ssl);
    mbedtls_ctr_drbg_init(&demo_ota_ssl_ctr_drbg);
    mbedtls_ssl_config_init(&demo_ota_ssl_conf);
    mbedtls_entropy_init(&demo_ota_ssl_entropy);

    if((ret = mbedtls_ctr_drbg_seed(&demo_ota_ssl_ctr_drbg, mbedtls_entropy_func, &demo_ota_ssl_entropy,
                                    (const unsigned char *) pers,
                                    strlen(pers))) != 0)
    {
        printf("mbedtls_ctr_drbg_seed returned %d\r\n", ret);
        goto exit;
    }

    if((ret = mbedtls_ssl_config_defaults(&demo_ota_ssl_conf,
                                          MBEDTLS_SSL_IS_CLIENT,
                                          MBEDTLS_SSL_TRANSPORT_STREAM,
                                          MBEDTLS_SSL_PRESET_DEFAULT)) != 0)
    {
        printf("mbedtls_ssl_config_defaults returned %d\r\n", ret);
        goto exit;
    }

    mbedtls_ssl_conf_authmode(&demo_ota_ssl_conf, MBEDTLS_SSL_VERIFY_OPTIONAL);
    mbedtls_ssl_conf_rng(&demo_ota_ssl_conf, mbedtls_ctr_drbg_random, &demo_ota_ssl_ctr_drbg);
#ifdef MBEDTLS_DEBUG_C
    mbedtls_debug_set_threshold(DEBUG_LEVEL);
    mbedtls_ssl_conf_dbg(&demo_ota_ssl_conf, demo_ota_ssl_debug, stdout);
#endif

    if((ret = mbedtls_ssl_setup(&demo_ota_ssl, &demo_ota_ssl_conf)) != 0)
    {
        printf("mbedtls_ssl_setup returned %d\r\n", ret);
        goto exit;
    }

    mbedtls_ssl_set_bio(&demo_ota_ssl, (void *)(unsigned long)skt, demo_ota_ssl_net_send, demo_ota_ssl_net_recv, NULL);

    while((ret = mbedtls_ssl_handshake(&demo_ota_ssl)) != 0)
    {
        if(ret != MBEDTLS_ERR_SSL_WANT_READ && ret != MBEDTLS_ERR_SSL_WANT_WRITE)
        {
            printf("mbedtls_ssl_handshake returned -0x%x\r\n", -ret);
            goto exit;
        }
    }

    return 0;

exit:
    mbedtls_entropy_free(&demo_ota_ssl_entropy);
    mbedtls_ssl_config_free(&demo_ota_ssl_conf);
    mbedtls_ctr_drbg_free(&demo_ota_ssl_ctr_drbg);
    mbedtls_ssl_free(&demo_ota_ssl);

    return -1;
}

static void demo_ota_ssl_close(void)
{
    mbedtls_ssl_close_notify(&demo_ota_ssl);

    mbedtls_entropy_free(&demo_ota_ssl_entropy);
    mbedtls_ssl_config_free(&demo_ota_ssl_conf);
    mbedtls_ctr_drbg_free(&demo_ota_ssl_ctr_drbg);
    mbedtls_ssl_free(&demo_ota_ssl);

    return;
}
#endif

static int demo_ota_recv(int skt, void *html, int len)
{
    int ret;
    fd_set read_set;
    struct timeval tv;

    FD_ZERO(&read_set);
    FD_SET(skt, &read_set);
    tv.tv_sec  = DEMO_TCP_NO_BLOCK_TIME;
    tv.tv_usec = 0;

    ret = select(skt + 1, &read_set, NULL, NULL, &tv);
    if (ret > 0)
    {
        if (FD_ISSET(skt, &read_set))
        {
#if DEMO_OTA_USE_HTTPS
            if (demo_ota_is_https)
                ret = mbedtls_ssl_read(&demo_ota_ssl, html, len);
            else
#endif
                ret = recv(skt, html, len, 0);

            if (0 == ret)
            {
                ret = -16;
            }
            else if (ret < 0)
            {

            }

            FD_CLR(skt, &read_set);
        }
        else
        {
            ret = -17;
        }
    }
    else
    {
        ret = -18;
    }

    return ret;
}

static int demo_ota_read_by_size(int skt_num, char *buf, u16 buf_max_size, u16 next_read_size)
{
	unsigned int total_to_read;
	int          cur_to_read;
	unsigned int cur_read;

    if (buf_max_size < next_read_size)
    {
        return -1;
	}

    cur_read      = 0;
    total_to_read = next_read_size;

    do
    {
        cur_to_read = demo_ota_recv(skt_num, buf + cur_read, total_to_read);
        if (0 >= cur_to_read)
            return cur_to_read;

        cur_read      += cur_to_read;
        total_to_read -= cur_to_read;
    } while(total_to_read > 0);

	return cur_read;
}

static void demo_ota_parse_line(char *line, u16 len)
{
    //printf("'%s'", line);

    if (sscanf(line, "Content-Length: %u", &demo_ota_content_len) == 1 ||
	    sscanf(line, "Content-length: %u", &demo_ota_content_len) == 1)
	{

	}

    return;
}

static int demo_ota_get_response(int skt, char *resp_buf, u32 resp_buf_size)
{
    int line_count, total_len = 0;
    int len;
    char *buf, *q;
    u8 ch;
    int ret;

    buf = tls_mem_alloc(DEMO_OTA_BUF_SIZE);
    if (!buf)
    {
        return -2;
    }

    line_count = 0;
    memset(resp_buf, 0, resp_buf_size);
    for(;;) {
        memset(buf, 0, DEMO_OTA_BUF_SIZE);
        q = buf;
        for(;;) {
            ret = demo_ota_recv(skt, &ch, 1);
            if (ret <= 0)
        	    break;
            if (ch == '\n')
                break;
            else if (ch != '\r') {
                if ((q - buf) < DEMO_OTA_BUF_SIZE - 1)
                    *q++ = ch;
            }
        }
        *q = '\0';

        /* test if last line */
        if (buf[0] == '\0')
        {
            *(resp_buf + total_len) = '\0';
            break;
        }

        if (line_count == 0) {
            *q++ = '\r';
            *q   = '\n';
            total_len  = q - buf + 1;
            memcpy(resp_buf, buf, total_len);
            //demo_ota_parse_line(buf, total_len);
        }
        else
        {
            *q++ = '\r';
            *q   = '\n';
            len = q - buf + 1;
            memcpy(resp_buf + total_len, buf, len);
            total_len += len;
            demo_ota_parse_line(buf, len);
        }
        line_count++;
    }

    tls_mem_free(buf);

	return total_len;
}

#if DEMO_TCP_CONNECT_NO_BLOCK
static int demo_ota_make_blocking(int sock)
{
	int val;

	val = fcntl(sock, F_GETFL, 0);
	if (fcntl(sock, F_SETFL, val & ~O_NONBLOCK) == -1)
	{
		return -1;
	}

	return 0;
}

static int demo_ota_make_no_blocking(int sock)
{
	int val;

	val = fcntl(sock, F_GETFL, 0);
	if (fcntl(sock, F_SETFL, val | O_NONBLOCK) == -1)
	{
		return -1;
	}

	return 0;
}

static int demo_ota_check_connection(int sock, int connect_res)
{
	struct timeval timeout = {DEMO_TCP_NO_BLOCK_TIME, 0};
	int result;
	fd_set fd_set;

	FD_ZERO(&fd_set);
	FD_SET(sock, &fd_set);

	if (connect_res < 0)
	{
		if ((EINPROGRESS == errno) || (0 == errno))
    	{
			result = select(sock + 1, NULL, &fd_set, NULL, &timeout);
			if (result < 0)
			{
				return -1;
			}
			else if (result == 0)
			{
				/* timeout */
				return -1;
			}
			else
			{
			    if (FD_ISSET(sock, &fd_set))
			    {
    				int valopt = 0;
    				socklen_t len = sizeof(valopt);
    				if (getsockopt(sock, SOL_SOCKET, SO_ERROR, (void *) &valopt, &len) < 0)
    				{
    					/* failed to read delayed error */
    					return -1;
    				}
    				else if (valopt)
    				{
    					/* delayed error = valopt */
    					return -1;
    				}
				}
				else
				{
                    return -1;
				}
			}
        }
        else
        {
            return -1;
        }
	}

	return 0;
}
#endif

#if DEMO_TCP_UNTIL_WRITEABLE
static int demo_ota_until_writeable(int socket)
{
	int result = -1;
	fd_set wd_set;
	struct timeval timeout = {DEMO_TCP_NO_BLOCK_TIME, 0};

	FD_ZERO(&wd_set);
	FD_SET(socket, &wd_set);

	result = select(socket + 1, NULL, &wd_set, NULL,&timeout);
	if (result <= 0)
	{
        result = -1;
	}
    else if (result > 0)
    {
    	if (FD_ISSET(socket, &wd_set))
    	{
    	    result = 0;
    	}
    	else
    	{
            result = -1;
    	}
	}

	return result;
}
#endif

static int demo_ota_connection(char *address, u16 port)
{
	struct hostent *hp;
	struct sockaddr_in server;
	int skt = -1;
	int ret = -1;

    hp = gethostbyname(address);
    if (hp == NULL )
	{
		return -11;
	}

	skt = socket(AF_INET, SOCK_STREAM, 0);
	if (skt < 0)
	{
		return -12;
	}

#if DEMO_TCP_CONNECT_NO_BLOCK
	ret = demo_ota_make_no_blocking(skt);
	if (ret < 0)
	{
		close(skt);
		return -13;
	}
#endif

	memset(&server, 0, sizeof(struct sockaddr_in));
	memcpy(&(server.sin_addr), hp->h_addr, hp->h_length);
	server.sin_family = AF_INET;
	server.sin_port = htons(port);
	ret = connect(skt, (struct sockaddr*)&server, sizeof(struct sockaddr_in));

#if DEMO_TCP_CONNECT_NO_BLOCK
	ret = demo_ota_check_connection(skt, ret);
    if (!ret)
        ret = demo_ota_make_blocking(skt);
#endif

#if DEMO_OTA_USE_HTTPS
    if (!ret && demo_ota_is_https)
    {
        ret = demo_ota_ssl_connect(skt);
    }
#endif

	if (ret < 0)
	{
		close(skt);
		skt = -14;
	}

	return skt;
}

static int demo_ota_parse_url(const char *url, char *address, u16 *port, char *media)
{
	u32 prefix_length = 7;
	char *from = NULL ;
	char *to = NULL;
	u32 i;
	int port_num_int;
	char next_char;

	if (strncmp(url, "http://", 7) != 0 
#if DEMO_OTA_USE_HTTPS
	&& strncmp(url, "https://", 8) != 0
#endif
	)
	{
		return -7;
	}

#if DEMO_OTA_USE_HTTPS
    if (strncmp(url, "https://", 8) == 0)
    {
        prefix_length = 8;
        demo_ota_is_https = 1;
    }
    else
    {
        demo_ota_is_https = 0;
    }
#endif

	from = (char *)&url[prefix_length];
	to = &address[0];
	for (i = 0; i < DEMO_OTA_ADDR_LEN; ++i)
	{
		if (*from == '\0' || *from == ':' || *from == '/')
		{
			*to = '\0';
			break;
		}
		*to++ = *from++;
	}

	if (i == DEMO_OTA_ADDR_LEN)
	{
		return -8;
	}

#if DEMO_OTA_USE_HTTPS
    if (demo_ota_is_https)
        *port = 443;
    else
#endif
 	    *port = 80;

	next_char = *from;
	if (next_char == ':')
	{

		if (sscanf(++from, "%d", &port_num_int) != 1)
		{
			return -9;
		}
		if (port_num_int < 1 || port_num_int > 65535)
		{
			return -10;
		}
		*port = port_num_int;
	}

    while (*from != '\0')
    {
        if (*from == '/')
        {
            strcpy(media, from);
            break;
        }
        from++;
    }

	return 0;
}

static int demo_ota_send_request(const char *url, u32 offset, char *content)
{
    int ret = -1;
    int len = 0;
    int skt;
    //char *pos;
    u16 port;
    char *address = NULL;
    char *media   =  NULL;
    char *html = NULL;
    int content_len = 0;

    address = tls_mem_alloc(DEMO_OTA_ADDR_LEN);
	if (!address)
	    return -2;

	media = tls_mem_alloc(DEMO_OTA_ADDR_LEN);
	if (!media)
	{
	    tls_mem_free(address);
	    return -2;
    }

	memset(address, 0, DEMO_OTA_ADDR_LEN);
	memset(media,   0, DEMO_OTA_ADDR_LEN);

	ret = demo_ota_parse_url(url, address, &port, media);
	if (ret)
	{
	    tls_mem_free(address);
	    tls_mem_free(media);
	    return ret;
    }

    html = tls_mem_alloc(DEMO_OTA_BUF_SIZE);
    if (!html)
    {
	    tls_mem_free(address);
	    tls_mem_free(media);
	    return -2;
    }

    skt = demo_ota_connection(address, port);
    if (skt < 0)
    {
        tls_mem_free(address);
	    tls_mem_free(media);
        tls_mem_free(html);
        return skt;
    }

#if 0
    if ((0 != strncmp(url, "http://", 7)) && (0 != strcmp(media, url)))
    {
        /* url׷ӵmediaĩβ滻ԭĵַ */
        pos = media + strlen(media);
        while (*pos != '/')
        {
            pos--;
        }
        strcpy(pos + 1, url);
    }
    else
    {
        strcpy(media, url);
    }
#endif

    memset(html, 0, DEMO_OTA_BUF_SIZE);
    if (content)
        len = sprintf(html, "POST %s HTTP/1.1\r\n", media);
    else
        len = sprintf(html, "GET %s HTTP/1.1\r\n", media);
    len += sprintf(html + len, "Host: %s\r\n", address);
    if (offset)
        len += sprintf(html + len, "Range: bytes=%u-\r\n", offset);
    len += sprintf(html + len, "User-Agent: %s\r\n", "w600-ota");
    if (content)
    {
        content_len = strlen(content);
        len += sprintf(html + len, "Content-Type: application/json; charset=utf-8\r\n");
        len += sprintf(html + len, "Content-Length: %d\r\n", content_len);
    }
    len += sprintf(html + len, "\r\n");
    if (content)
    {
        strcpy(html + len, content);
        len += content_len;
    }

    tls_mem_free(address);
    tls_mem_free(media);

    //printf("skt %d, '%s'\r\n", skt, html);

#if DEMO_TCP_UNTIL_WRITEABLE
    ret = demo_ota_until_writeable(skt);
    if (0 == ret)
    {
#endif
#if DEMO_OTA_USE_HTTPS
        if (demo_ota_is_https)
            ret = mbedtls_ssl_write(&demo_ota_ssl, html, len);
        else
#endif
    	    ret = send(skt, html, len, 0);

        if (ret != len)
        {
#if DEMO_OTA_USE_HTTPS
            if (demo_ota_is_https)
                demo_ota_ssl_close();
#endif
            close(skt);
            tls_mem_free(html);
            return -15;
        }
#if DEMO_TCP_UNTIL_WRITEABLE
    }
    else
    {
        ret = -21;
    }
#endif

    tls_mem_free(html);

    return skt;
}

static int demo_ota_entry(const char *fw_url)
{
    int ret = -1;
    int skt;
    char *html = NULL;
    u32 response_code = 0;
    u32 read = 0;
    u32 total = 0;
	u32 seg_len = 0;
	u32 offset = 0; /* ϵ */

    skt = demo_ota_send_request(fw_url, offset, NULL);
    if (skt < 0)
        return skt;

    html = tls_mem_alloc(DEMO_OTA_BUF_SIZE);
    if (!html)
    {
        close(skt);
        return -2;
    }

    demo_ota_content_len = 0;
    memset(html, 0, DEMO_OTA_BUF_SIZE);
    ret = demo_ota_get_response(skt, html, DEMO_OTA_BUF_SIZE);
    if (ret > 0)
    {
        if (sscanf(html, "%*s%u", &response_code) != 1)
    	{
    		printf("ota no response code in line\n");
    		ret = -3;
    		goto fail;
    	}

    	if (((0 == offset) && (200 != response_code)) ||
    	    ((0 != offset) && (206 != response_code)))
    	{
    		printf("ota response code %u not success\n", response_code);
    		ret = -4;
            goto fail;
    	}

    	if (demo_ota_content_len >= CODE_UPD_AREA_LEN)
    	{
    		printf("firmware size %u too big.\n", demo_ota_content_len);
    		ret = -20;
            goto fail;
    	}

    	printf("firmwire size is %u bytes\r\n", demo_ota_content_len);

    	demo_ota_session_id = tls_fwup_enter(TLS_FWUP_IMAGE_SRC_WEB);

        while (demo_ota_content_len > total)
        {
            read = demo_ota_content_len - total;
            if (read > DEMO_OTA_BUF_SIZE)
                read = DEMO_OTA_BUF_SIZE;

            ret = demo_ota_read_by_size(skt, html, DEMO_OTA_BUF_SIZE, read);
            if (ret > 0)
            {
                total += ret;
                ret = tls_fwup_request_sync(demo_ota_session_id, (u8 *)html, ret);
                if (TLS_FWUP_STATUS_OK != ret)
                {
                    tls_fwup_exit(demo_ota_session_id);
                    ret = -19;
                    goto fail;
                }

                seg_len += read;
                if (seg_len >= 4096)
                {
                    seg_len = seg_len % 4096;
                    demo_ota_download_progress(total + offset, demo_ota_content_len + offset);
                }
            }
            else
            {
                ret = -6;
                break;
            }
        }

        tls_fwup_exit(demo_ota_session_id);
    }
    else
    {
        ret = -5;
    }

fail:
#if DEMO_OTA_USE_HTTPS
    if (demo_ota_is_https)
        demo_ota_ssl_close();
#endif
    close(skt);
    tls_mem_free(html);

    return ret;
}

static void demo_task_entry(void *sdata)
{
    int ret;

    tls_wifi_connect(DEMO_SSID, strlen(DEMO_SSID), DEMO_PWD, strlen(DEMO_PWD));

    struct tls_ethif *ethif = tls_netif_get_ethif();
    while (!ethif->status) tls_os_time_delay(100);

    printf("wifi connected, start ota...\r\n");

    ret = demo_ota_entry(DEMO_URL);

    printf("ota failed = %d\r\n", ret); /* w600ǳɹԶλߵ˵otaʧ... */

	for( ; ; )
	{
        tls_os_time_delay(999999);
	}
}

void UserMain(void)
{
	printf("\n http(s) ota demo\n");

    tls_os_task_create(NULL, "demo_task", demo_task_entry, NULL, (void *)demo_task_stk, DEMO_TASK_STK_SIZE, DEMO_TASK_PRIO, 0);
}

