#include <string.h>
#include "wm_type_def.h"
#include "wm_ieee80211.h"
#include "tls_common.h"
#include "wm_wifi.h"
#include "wm_regs.h"

static u8  tls_wifi_current_channel = 0;
static u32 tls_wifi_capture_cnt  = 0;
static u32 tls_wifi_capture_time = 0;
static u8  tls_wifi_capture_filter_src_mac[ETH_ALEN] = {0};
static u8  tls_wifi_capture_filter_dst_mac[ETH_ALEN] = {0};
static int tls_wifi_capture_filter_type    = -1;
static u8  tls_wifi_capture_filter_channel = 1;
static tls_os_timer_t *tls_wifi_capture_filter_tmier = NULL;

static char *tls_wifi_capture_get_protocol(u16 fc)
{
    if (ieee80211_is_data(fc))
    {
        if ((fc & IEEE80211_FCTL_STYPE) == IEEE80211_STYPE_DATA)
            return "802.11 Data";
        else if ((fc & IEEE80211_FCTL_STYPE) == IEEE80211_STYPE_DATA_CFACK)
            return "Data CF-Ack";
        else if ((fc & IEEE80211_FCTL_STYPE) == IEEE80211_STYPE_DATA_CFPOLL)
            return "Data CF-Poll";
        else if ((fc & IEEE80211_FCTL_STYPE) == IEEE80211_STYPE_DATA_CFACKPOLL)
            return "Data CF-Ack-Poll";
        else if ((fc & IEEE80211_FCTL_STYPE) == IEEE80211_STYPE_NULLFUNC)
            return "Null-Func";
        else if ((fc & IEEE80211_FCTL_STYPE) == IEEE80211_STYPE_CFACK)
            return "CF-Ack";
        else if ((fc & IEEE80211_FCTL_STYPE) == IEEE80211_STYPE_CFPOLL)
            return "CF-Poll";
        else if ((fc & IEEE80211_FCTL_STYPE) == IEEE80211_STYPE_CFACKPOLL)
            return "CF-Ack-Poll";
        else if ((fc & IEEE80211_FCTL_STYPE) == IEEE80211_STYPE_QOS_DATA)
            return "Qos Data";
        else if ((fc & IEEE80211_FCTL_STYPE) == IEEE80211_STYPE_QOS_DATA_CFACK)
            return "Qos CF-Ack";
        else if ((fc & IEEE80211_FCTL_STYPE) == IEEE80211_STYPE_QOS_DATA_CFPOLL)
            return "Qos CF-Poll";
        else if ((fc & IEEE80211_FCTL_STYPE) == IEEE80211_STYPE_QOS_DATA_CFACKPOLL)
            return "Qos CF-Ack-Poll";
        else if ((fc & IEEE80211_FCTL_STYPE) == IEEE80211_STYPE_QOS_NULLFUNC)
            return "Qos Null-Func";
        else if ((fc & IEEE80211_FCTL_STYPE) == IEEE80211_STYPE_QOS_CFACK)
            return "Qos CF-Ack";
        else if ((fc & IEEE80211_FCTL_STYPE) == IEEE80211_STYPE_QOS_CFPOLL)
            return "Qos CF-Poll";
        else if ((fc & IEEE80211_FCTL_STYPE) == IEEE80211_STYPE_QOS_CFACKPOLL)
            return "Qos CF-Ack-Poll";
        else
            return "802.11 Data";/* crc error */
    }
    else if (ieee80211_is_mgmt(fc))
    {
        if ((fc & IEEE80211_FCTL_STYPE) == IEEE80211_STYPE_ASSOC_REQ)
            return "802.11 Asso Req";
        else if ((fc & IEEE80211_FCTL_STYPE) == IEEE80211_STYPE_ASSOC_RESP)
            return "802.11 Asso Rsp";
        else if ((fc & IEEE80211_FCTL_STYPE) == IEEE80211_STYPE_REASSOC_REQ)
            return "802.11 Resso Req";
        else if ((fc & IEEE80211_FCTL_STYPE) == IEEE80211_STYPE_REASSOC_RESP)
            return "802.11 Resso Rsp";
        else if ((fc & IEEE80211_FCTL_STYPE) == IEEE80211_STYPE_PROBE_REQ)
            return "802.11 Probe Req";
        else if ((fc & IEEE80211_FCTL_STYPE) == IEEE80211_STYPE_PROBE_RESP)
            return "802.11 Probe Rsp";
        else if ((fc & IEEE80211_FCTL_STYPE) == IEEE80211_STYPE_BEACON)
            return "802.11 Beacon";
        else if ((fc & IEEE80211_FCTL_STYPE) == IEEE80211_STYPE_ATIM)
            return "802.11 Atim";
        else if ((fc & IEEE80211_FCTL_STYPE) == IEEE80211_STYPE_DISASSOC)
            return "802.11 Disasso";
        else if ((fc & IEEE80211_FCTL_STYPE) == IEEE80211_STYPE_AUTH)
            return "802.11 Auth";
        else if ((fc & IEEE80211_FCTL_STYPE) == IEEE80211_STYPE_DEAUTH)
            return "802.11 Deauth";
        else if ((fc & IEEE80211_FCTL_STYPE) == IEEE80211_STYPE_ACTION)
            return "802.11 Action";
        else
            return "802.11 Management";/* crc error */
    }
    else if (ieee80211_is_ctl(fc))
    {
        if ((fc & IEEE80211_FCTL_STYPE) == IEEE80211_STYPE_BACK_REQ)
            return "802.11 Back Req";
        else if ((fc & IEEE80211_FCTL_STYPE) == IEEE80211_STYPE_BACK)
            return "802.11 Back";
        else if ((fc & IEEE80211_FCTL_STYPE) == IEEE80211_STYPE_PSPOLL)
            return "802.11 Ps-Poll";
        else if ((fc & IEEE80211_FCTL_STYPE) == IEEE80211_STYPE_RTS)
            return "802.11 Rts";
        else if ((fc & IEEE80211_FCTL_STYPE) == IEEE80211_STYPE_CTS)
            return "802.11 Cts";
        else if ((fc & IEEE80211_FCTL_STYPE) == IEEE80211_STYPE_ACK)
            return "802.11 Ack";
        else if ((fc & IEEE80211_FCTL_STYPE) == IEEE80211_STYPE_CFEND)
            return "802.11 CF-End";
        else if ((fc & IEEE80211_FCTL_STYPE) == IEEE80211_STYPE_CFENDACK)
            return "802.11 CF-End-Ack";
        else
            return "802.11 Control";/* crc error */
    }

    return "Unknown-type";/* crc error */
}

static void tls_wifi_capture_get_flag(struct ieee80211_hdr *hdr,
                                      struct tls_wifi_ext_t *ext,
                                      char *flag)
{
    u8 i = 0;
    struct ieee80211_mgmt *mgmt;

    if (ieee80211_is_mgmt(hdr->frame_control))
        flag[i++] = '*';
    else if (ieee80211_is_ctl(hdr->frame_control))
        flag[i++] = '#';

    if (ext->crc)
        flag[i++] = 'C';

    if (ext->aggr)
        flag[i++] = 'A';

    if (ieee80211_is_data(hdr->frame_control) &&
        ieee80211_has_protected(hdr->frame_control))
        flag[i++] = 'W';

    if (ieee80211_is_mgmt(hdr->frame_control))
    {
        mgmt= (struct ieee80211_mgmt *)hdr;
        switch (hdr->frame_control & IEEE80211_FCTL_STYPE)
        {
            case IEEE80211_STYPE_ASSOC_REQ:
            {
                if (mgmt->u.assoc_req.capab_info & BIT(4))
                    flag[i++] = 'P';
                break;
            }
            case IEEE80211_STYPE_ASSOC_RESP:
            {
                if (mgmt->u.assoc_resp.capab_info & BIT(4))
                    flag[i++] = 'P';
                break;
            }
            case IEEE80211_STYPE_REASSOC_RESP:
            {
                if (mgmt->u.reassoc_resp.capab_info & BIT(4))
                    flag[i++] = 'P';
                break;
            }
            case IEEE80211_STYPE_REASSOC_REQ:
            {
                if (mgmt->u.reassoc_req.capab_info & BIT(4))
                    flag[i++] = 'P';
                break;
            }
            case IEEE80211_STYPE_BEACON:
            {
                if (mgmt->u.beacon.capab_info & BIT(4))
                    flag[i++] = 'P';
                break;
            }
            case IEEE80211_STYPE_PROBE_RESP:
            {
                if (mgmt->u.probe_resp.capab_info & BIT(4))
                    flag[i++] = 'P';
                break;
            }
            default:
            {
                break;
            }
        }
    }

    if (ieee80211_has_retry(hdr->frame_control))
        flag[i++] = '+';

    flag[i] = '\0';

    return;
}

static void tls_wifi_capture_get_summary(struct ieee80211_hdr *hdr,
                                         u32 data_len,
                                         struct tls_wifi_ext_t *ext,
                                         char *summary)
{
    u8 len;

    if (ieee80211_has_tods(hdr->frame_control))
        len = sprintf(summary, "FC=T, SN=%hu, FN=%hhu", hdr->seq_ctrl >> 4, hdr->seq_ctrl & 0xF);
    else if (ieee80211_has_fromds(hdr->frame_control))
        len = sprintf(summary, "FC=F, SN=%hu, FN=%hhu", hdr->seq_ctrl >> 4, hdr->seq_ctrl & 0xF);
    else
        len = sprintf(summary, "SN=%hu, FN=%hhu", hdr->seq_ctrl >> 4, hdr->seq_ctrl & 0xF);

    len += sprintf(summary + len, ", OR_RSSI=%hhu", ext->rssi);

    sprintf(summary + len, "...");

    return;
}

const static float tls_wifi_11n_rate_table[17][4] = {
    {  6.50,   7.20,  13.50,  15.00},/* mcs0 */
    { 13.00,  14.40,  27.00,  30.00},/* mcs1 */
    { 19.50,  21.70,  40.50,  45.00},/* mcs2 */
    { 26.00,  28.90,  54.00,  60.00},/* mcs3 */
    { 39.00,  43.30,  81.00,  90.00},/* mcs4 */
    { 52.00,  57.80, 108.00, 120.00},/* mcs5 */
    { 58.50,  65.00, 121.50, 135.00},/* mcs6 */
    { 65.00,  72.20, 135.00, 150.00},/* mcs7 */
    { 13.00,  14.40,  27.00,  30.00},/* mcs8 */
    { 26.00,  28.90,  54.00,  60.00},/* mcs9 */
    { 39.00,  43.30,  81.00,  90.00},/* mcs10 */
    { 52.00,  57.80, 108.00, 120.00},/* mcs11 */
    { 78.00,  86.70, 162.00, 180.00},/* mcs12 */
    {104.00, 115.60, 216.00, 240.00},/* mcs13 */
    {117.00, 130.00, 243.00, 270.00},/* mcs14 */
    {130.00, 144.40, 270.00, 300.00},/* mcs15 */
    {260.00, 288.90, 540.00, 600.00}/* mcs31 */
};

static float tls_wifi_capture_get_rate(struct tls_wifi_ext_t *ext)
{
    float rate;
    u8 x = 0, y = 0;

    if (ext->rate <= 0x6C)
    {
        rate = (double)ext->rate / 2;
    }
    else
    {
        if (ext->cbw)/* 40Mhz */
        {
            if (ext->sgi)/* shoft gi */
            {
                y = 3;
            }
            else
            {
                y = 2;
            }
        }
        else/* 20Mhz */
        {
            if (ext->sgi)/* shoft gi */
            {
                y = 1;
            }
            else
            {
                y = 0;
            }
        }

        if (ext->rate < 0xA0)
            x = ext->rate - 0x80;
        else
            x = 16;

        rate = tls_wifi_11n_rate_table[x][y];
    }

    return rate;
}

static int tls_wifi_capture_get_signal(u8 rssi)
{
    int signal;

    signal = 110 - ((int)rssi / 2);

	if (signal <= 30)
	{
		signal = 100;
	}
	else if (signal >= 80)
	{
		signal = 1;
	}
	else
	{
		signal = (int) ((double) (1 + 99 * 8 / 5) - (double) signal * 99 / 50);
	}

	if (signal > 100)
	    signal = 100;

    return signal;
}

static bool tls_wifi_capture_is_drop(struct ieee80211_hdr *hdr, u32 data_len)
{
    u8   *sa;
    u8   *da;

    if (tls_wifi_capture_filter_type >= 0)
    {
        if ((hdr->frame_control &
             (IEEE80211_FCTL_FTYPE | IEEE80211_FCTL_STYPE)) !=
            tls_wifi_capture_filter_type)
        {
            //printf("%d drop: type %x\n", __LINE__, hdr->frame_control & (IEEE80211_FCTL_FTYPE | IEEE80211_FCTL_STYPE));
            return true;
        }
    }

    sa = ieee80211_get_SA(hdr);
    da = ieee80211_get_DA(hdr);

    if (!is_zero_ether_addr(tls_wifi_capture_filter_src_mac))
    {
        if (0 != compare_ether_addr(sa, tls_wifi_capture_filter_src_mac))
        {
            //printf("%d drop: src "MACSTR"\n", __LINE__, MAC2STR(sa));
            return true;
        }
    }

    if (!is_zero_ether_addr(tls_wifi_capture_filter_dst_mac))
    {
        if (0 != compare_ether_addr(da, tls_wifi_capture_filter_dst_mac))
        {
            //printf("%d drop: dst "MACSTR"\n", __LINE__, MAC2STR(da));
            return true;
        }
    }

    return false;
}

static void tls_wifi_capture_callback(u8* data, u32 data_len, struct tls_wifi_ext_t *ext)
{
    struct ieee80211_hdr *hdr;
    char *protocol;
    u8   *sa;
    u8   *da;
    u8   *bssid;
    char summary[80];
    char flags[10];
    u8   channel;
    int  signal;
    u32  timestap;
    float rate;

    hdr = (struct ieee80211_hdr *)data;

    if (tls_wifi_capture_is_drop(hdr, data_len))
        return;

    protocol = tls_wifi_capture_get_protocol(hdr->frame_control);

    sa = ieee80211_get_SA(hdr);
    da = ieee80211_get_DA(hdr);
    bssid = (u8 *)get_hdr_bssid(hdr, data_len);

    tls_wifi_capture_get_flag(hdr, ext, flags);

    channel = tls_wifi_current_channel;

    rate = tls_wifi_capture_get_rate(ext);

    signal = tls_wifi_capture_get_signal(ext->rssi);

    tls_wifi_capture_get_summary(hdr, data_len, ext, summary);

    timestap = (tls_os_get_time() - tls_wifi_capture_time) * 10;

    tls_wifi_capture_cnt++;

    printf("%-10u  %-20s  "MACSTR"  "MACSTR"  "MACSTR"  %-4u  %-5s  %7hhu  "
           "%5d%%  %4gMbps  %7ums  %-7s\r\n", tls_wifi_capture_cnt, protocol,
           MAC2STR(sa), MAC2STR(da), MAC2STR(bssid), data_len, flags, channel,
           signal, rate, timestap, summary);

    return;
}

static void tls_wifi_capture_filter_swchn_callback(void *ptmr, void *parg)
{
    if (++tls_wifi_current_channel > 14)
        tls_wifi_current_channel = 1;

    tls_wifi_change_chanel(tls_wifi_current_channel - 1);
}

void tls_wifi_capture_filter(u8 channel, u8 *src_mac, u8 *dst_mac, int type)
{
    tls_wifi_capture_filter_channel = channel;

    if (src_mac)
        memcpy(tls_wifi_capture_filter_src_mac, src_mac, ETH_ALEN);
    else
        memset(tls_wifi_capture_filter_src_mac, 0, ETH_ALEN);

    if (dst_mac)
        memcpy(tls_wifi_capture_filter_dst_mac, dst_mac, ETH_ALEN);
    else
        memset(tls_wifi_capture_filter_dst_mac, 0, ETH_ALEN);

    if (type >= 0)
        tls_wifi_capture_filter_type = type;

    //printf("ch %hhu, src "MACSTR", dst "MACSTR", type %d\n", channel, MAC2STR(src_mac), MAC2STR(dst_mac), type);

    return;
}

void tls_wifi_capture_start(void)
{
    if (tls_wifi_capture_filter_channel != 0xFF)
    {
        tls_wifi_change_chanel(tls_wifi_capture_filter_channel - 1);
        tls_wifi_current_channel = tls_wifi_capture_filter_channel;
    }
    else
    {
        if (NULL == tls_wifi_capture_filter_tmier)
            tls_os_timer_create(&tls_wifi_capture_filter_tmier,
                                tls_wifi_capture_filter_swchn_callback,
                                NULL, 10, true, NULL);
    }

    tls_wifi_set_listen_mode(1);

    printf("Packet      Protocol              Source             Destination        "
           "BSSID              Size  Flags  Channel  Signal  DataRate  "
           "DeltaTime  Summary\r\n");

    tls_wifi_capture_cnt  = 0;
    tls_wifi_capture_time = tls_os_get_time();

    tls_wifi_data_ext_recv_cb_register(tls_wifi_capture_callback);

    if (tls_wifi_capture_filter_channel == 0xFF)
        tls_os_timer_start(tls_wifi_capture_filter_tmier);

    return;
}

void tls_wifi_capture_stop(void)
{
    if (tls_wifi_capture_filter_tmier)
        tls_os_timer_stop(tls_wifi_capture_filter_tmier);

    tls_wifi_data_ext_recv_cb_register(NULL);

    tls_wifi_set_listen_mode(0);

    return;
}


void UserMain(void)
{
    u8 src_mac[6] = {0};
    u8 dst_mac[6] = {0};
    u8 channel    =  1;
    int type      = -1;
    u8 action     =  1;

    if (action)
    {
        tls_wifi_capture_filter(channel, src_mac, dst_mac, type);
        tls_wifi_capture_start();
    }
    else
    {
        tls_wifi_capture_stop();
    }

    return 0;
}
