/*-----------------------------------------------------------------------*/
/* Low level disk I/O module skeleton for FatFs     (C)ChaN, 2016        */
/*-----------------------------------------------------------------------*/
/* If a working storage control module is available, it should be        */
/* attached to the FatFs via a glue function rather than modifying it.   */
/* This is an example of glue functions to attach various exsisting      */
/* storage control modules to the FatFs module with a defined API.       */
/*-----------------------------------------------------------------------*/

#include "ff.h"			/* Obtains integer types */
#include "diskio.h"		/* Declarations of disk functions */

#include "wm_include.h"
#include "wm_flash_map.h"
#include "wm_internal_flash.h"
#include "wm_flash.h"

#include "wm_sd_driver.h"

/* Definitions of physical drive number for each drive */
#define DEV_SD		0	/* Example: Map Ramdisk to physical drive 0 */
#define DEV_SPI_INTERVAL	1
#define DEV_SPI_EXTERNAL	2

#define SD_CARD_BLOCK_SIZE         (4096)
#define SD_CARD_SECTOR_SIZE        (FF_MAX_SS)

#define INTERVAL_FLS_BASE               (CODE_UPD_START_ADDR)//(USER_ADDR_START)
#define INTERVAL_FLS_LEN                (CODE_UPD_AREA_LEN + USER_AREA_LEN)
#define INTERVAL_FLS_BLOCK_SIZE         (1)
#define INTERVAL_LFS_SECTOR_SIZE        (FF_MAX_SS)

#define EXTERNAL_FLS_BASE               (0x0)
#define EXTERNAL_FLS_LEN                (0x100000000)//4G bytes
#define EXTERNAL_FLS_BLOCK_SIZE         (1)
#define EXTERNAL_LFS_SECTOR_SIZE        (FF_MAX_SS)

/*-----------------------------------------------------------------------*/
/* Get Drive Status                                                      */
/*-----------------------------------------------------------------------*/

DSTATUS disk_status (
	BYTE pdrv		/* Physical drive nmuber to identify the drive */
)
{
	//DSTATUS stat;
	//int result;

	switch (pdrv) {
	case DEV_SD :
    case DEV_SPI_INTERVAL :
	case DEV_SPI_EXTERNAL :
		//result = RES_OK;

		// translate the reslut code here

		return RES_OK;
	}
	return STA_NOINIT;
}



/*-----------------------------------------------------------------------*/
/* Inidialize a Drive                                                    */
/*-----------------------------------------------------------------------*/

DSTATUS disk_initialize (
	BYTE pdrv				/* Physical drive nmuber to identify the drive */
)
{
	DSTATUS stat;
	int result;

	switch (pdrv) {
	case DEV_SD :
		#if 1
		result = SD_Init();

		// translate the reslut code here
		if (result)
			stat = RES_ERROR;
		else
		#endif
			stat = RES_OK;

		return stat;

    case DEV_SPI_INTERVAL :
	case DEV_SPI_EXTERNAL :
	    return RES_OK;
	}
	return STA_NOINIT;
}



/*-----------------------------------------------------------------------*/
/* Read Sector(s)                                                        */
/*-----------------------------------------------------------------------*/

DRESULT disk_read (
	BYTE pdrv,		/* Physical drive nmuber to identify the drive */
	BYTE *buff,		/* Data buffer to store read data */
	DWORD sector,	/* Start sector in LBA */
	UINT count		/* Number of sectors to read */
)
{
	DRESULT res;
	int result;

	switch (pdrv) {
	case DEV_SD :
		// translate the arguments here

		if (count > 1)
			result = SD_ReadMultiBlock(sector, buff, count);
		else
			result = SD_ReadSingleBlock(sector, buff);

		// translate the reslut code here
		if (result)
			res = RES_ERROR;
		else
			res = RES_OK;

		return res;

    case DEV_SPI_INTERVAL :
        result = tls_fls_read(INTERVAL_FLS_BASE + (sector * INTERVAL_LFS_SECTOR_SIZE),
                              buff, INTERVAL_LFS_SECTOR_SIZE * count);
		res = (TLS_FLS_STATUS_OK == result) ? RES_OK : RES_ERROR;
        return res;

    case DEV_SPI_EXTERNAL :
        result = tls_spifls_read(EXTERNAL_FLS_BASE + (sector * EXTERNAL_LFS_SECTOR_SIZE),
                                 buff, EXTERNAL_LFS_SECTOR_SIZE * count);
        res = (TLS_FLS_STATUS_OK == result) ? RES_OK : RES_ERROR;
        return res;
	}

	return RES_PARERR;
}



/*-----------------------------------------------------------------------*/
/* Write Sector(s)                                                       */
/*-----------------------------------------------------------------------*/

#if FF_FS_READONLY == 0

DRESULT disk_write (
	BYTE pdrv,			/* Physical drive nmuber to identify the drive */
	const BYTE *buff,	/* Data to be written */
	DWORD sector,		/* Start sector in LBA */
	UINT count			/* Number of sectors to write */
)
{
	DRESULT res;
	int result;

	switch (pdrv) {
	case DEV_SD :
		// translate the arguments here

		if (count > 1)
			result = SD_WriteMultiBlock(sector, buff, count);
		else
			result = SD_WriteSingleBlock(sector, buff);

		// translate the reslut code here
		if (result)
			res = RES_ERROR;
		else
			res = RES_OK;

		return res;

    case DEV_SPI_INTERVAL :
        result = tls_fls_write(INTERVAL_FLS_BASE + (sector * INTERVAL_LFS_SECTOR_SIZE),
                               (u8 *)buff, INTERVAL_LFS_SECTOR_SIZE * count);
		res = (TLS_FLS_STATUS_OK == result) ? RES_OK : RES_ERROR;
        return res;

    case DEV_SPI_EXTERNAL :
        result = tls_spifls_write(EXTERNAL_FLS_BASE + (sector * EXTERNAL_LFS_SECTOR_SIZE),
                                  (u8 *)buff, EXTERNAL_LFS_SECTOR_SIZE * count);
		res = (TLS_FLS_STATUS_OK == result) ? RES_OK : RES_ERROR;
        return res;
	}

	return RES_PARERR;
}

#endif


/*-----------------------------------------------------------------------*/
/* Miscellaneous Functions                                               */
/*-----------------------------------------------------------------------*/

DRESULT disk_ioctl (
	BYTE pdrv,		/* Physical drive nmuber (0..) */
	BYTE cmd,		/* Control code */
	void *buff		/* Buffer to send/receive control data */
)
{
	DRESULT res;
	//int result;

	switch (pdrv) {
    	case DEV_SD :
        {
    		// Process of the command for the RAM drive
    		switch (cmd) {
    			case GET_BLOCK_SIZE:
    				*(DWORD *)buff = SD_CARD_BLOCK_SIZE;
    				return RES_OK;
    			case GET_SECTOR_SIZE:
    				*(DWORD *)buff = SD_CARD_SECTOR_SIZE;
    				return RES_OK;
    			case GET_SECTOR_COUNT:
    				*(DWORD *)buff = SD_GetCapacity() / SD_CARD_SECTOR_SIZE;
    				return RES_OK;
    			case MMC_GET_CSD:
    				SD_GetCSD(buff);
    				return RES_OK;
    			case MMC_GET_CID:
    				SD_GetCID(buff);
    				return RES_OK;
    			case MMC_GET_TYPE:
    				*(UINT *)buff = SD_GetCardType();
    				return RES_OK;
    		}
        }
        case DEV_SPI_INTERVAL :
    	{
    	    res = RES_OK;
            switch (cmd) {
                case CTRL_SYNC:

                    break;
                case GET_SECTOR_COUNT:
                    *(DWORD * )buff = (INTERVAL_FLS_LEN / INTERVAL_LFS_SECTOR_SIZE);//INSIDE_FLS_PAGE_SIZE;
                    break;
                case GET_SECTOR_SIZE:
                    *(DWORD * )buff = INTERVAL_LFS_SECTOR_SIZE;
                    break;
                case GET_BLOCK_SIZE:
                    *(DWORD * )buff = INTERVAL_FLS_BLOCK_SIZE;
                    break;
                default:
                    res = RES_PARERR;
                    break;
            }
            return res;
        }
        case DEV_SPI_EXTERNAL:
        {
    	    res = RES_OK;
            switch (cmd) {
                case CTRL_SYNC:

                    break;
                case GET_SECTOR_COUNT:
                    *(DWORD * )buff = (EXTERNAL_FLS_LEN / EXTERNAL_LFS_SECTOR_SIZE);
                    break;
                case GET_SECTOR_SIZE:
                    *(DWORD * )buff = EXTERNAL_LFS_SECTOR_SIZE;
                    break;
                case GET_BLOCK_SIZE:
                    *(DWORD * )buff = EXTERNAL_FLS_BLOCK_SIZE;
                    break;
                default:
                    res = RES_PARERR;
                    break;
            }
            return res;
        }
	}

    //printf("disk_ioctl, error cmd = %d\r\n", cmd);
	return RES_PARERR;
}

DWORD get_fattime (void)
{
	return 0;//tls_os_get_time();
}

